/**
 * Countdown Timer Pro - Premium Customizer Panel
 * 
 * Provides a comprehensive visual interface for customizing the Countdown Timer Pro plugin.
 * Features include preset management, real-time preview, code generation, and export functionality.
 * 
 * @version 1.0.0
 * @author Xtreme Code
 */

(function($) {
    'use strict';
    
    // Premium Presets
    const PRESETS = {
        'premium-base': {
            name: 'Premium Base',
            timeType: 'countdown',
            width: 900,
            height: 450,
            fontSize: 56,
            circleThickness: 14,
            colors: {
                circle: '#667eea',
                circleBackground: '#1e293b',
                text: '#ffffff'
            },
            fontFamily: "'Segoe UI', Tahoma, sans-serif",
            fontWeight: '700',
            labelFontWeight: '500',
            showDays: true,
            showHours: true,
            showMinutes: true,
            showSeconds: true,
            multicolor: {
                enabled: false
            },
            gradientWave: {
                enabled: false,
                colors: ['#667eea', '#764ba2', '#f093fb', '#4facfe'],
                speed: 1.0
            }
        },
        'luxury-gold': {
            name: 'Luxury Gold',
            timeType: 'countdown',
            width: 900,
            height: 450,
            fontSize: 56,
            circleThickness: 12,
            colors: {
                circle: '#ffd700',
                circleBackground: '#1a1a1a',
                text: '#ffffff'
            },
            fontFamily: "Georgia, serif",
            fontWeight: '700',
            labelFontWeight: '400',
            showDays: true,
            showHours: true,
            showMinutes: true,
            showSeconds: true,
            multicolor: {
                enabled: true,
                colors: ['#ffd700', '#ffed4e', '#f4d03f', '#ffc107']
            },
            gradientWave: {
                enabled: false,
                colors: ['#ffd700', '#ffed4e', '#f4d03f', '#ffc107'],
                speed: 1.0
            }
        },
        'dark-neon': {
            name: 'Dark Neon',
            timeType: 'countdown',
            width: 900,
            height: 450,
            fontSize: 56,
            circleThickness: 8,
            colors: {
                circle: '#00ffff',
                circleBackground: '#0a0a0a',
                text: '#ffffff'
            },
            fontFamily: "'System Mono', monospace",
            fontWeight: '700',
            labelFontWeight: '400',
            showDays: true,
            showHours: true,
            showMinutes: true,
            showSeconds: true,
            multicolor: {
                enabled: true,
                colors: ['#00ffff', '#ff00ff', '#00ff00', '#ff00ff']
            },
            gradientWave: {
                enabled: false,
                colors: ['#00ffff', '#ff00ff', '#00ff00', '#ffff00'],
                speed: 1.0
            }
        },
        'glassmorphism': {
            name: 'Glassmorphism',
            timeType: 'countdown',
            width: 900,
            height: 450,
            fontSize: 52,
            circleThickness: 6,
            colors: {
                circle: '#667eea',
                circleBackground: '#1e293b',
                text: '#ffffff'
            },
            fontFamily: "'Segoe UI', Tahoma, sans-serif",
            fontWeight: '600',
            labelFontWeight: '400',
            showDays: true,
            showHours: true,
            showMinutes: true,
            showSeconds: true,
            multicolor: {
                enabled: false
            },
            gradientWave: {
                enabled: false,
                colors: ['#667eea', '#818cf8', '#a78bfa', '#c084fc'],
                speed: 1.0
            }
        },
        'minimalist': {
            name: 'Minimalist',
            timeType: 'countdown',
            width: 800,
            height: 400,
            fontSize: 48,
            circleThickness: 4,
            colors: {
                circle: '#4a5568',
                circleBackground: '#1a202c',
                text: '#ffffff'
            },
            fontFamily: "'System Sans', -apple-system, sans-serif",
            fontWeight: '400',
            labelFontWeight: '300',
            showDays: true,
            showHours: true,
            showMinutes: true,
            showSeconds: true,
            multicolor: {
                enabled: false
            },
            gradientWave: {
                enabled: false,
                colors: ['#4a5568', '#718096', '#a0aec0', '#cbd5e0'],
                speed: 1.0
            }
        },
        'cyberpunk': {
            name: 'Cyberpunk',
            timeType: 'countdown',
            width: 900,
            height: 450,
            fontSize: 56,
            circleThickness: 10,
            colors: {
                circle: '#ff00ff',
                circleBackground: '#1a0033',
                text: '#ffffff'
            },
            fontFamily: "'System Mono', monospace",
            fontWeight: '700',
            labelFontWeight: '400',
            showDays: true,
            showHours: true,
            showMinutes: true,
            showSeconds: true,
            multicolor: {
                enabled: true,
                colors: ['#ff00ff', '#00ffff', '#ffff00', '#ff0080']
            },
            gradientWave: {
                enabled: false,
                colors: ['#ff00ff', '#00ffff', '#ffff00', '#ff0080'],
                speed: 1.0
            }
        },
        'elegant-classic': {
            name: 'Elegant Classic',
            timeType: 'countdown',
            width: 900,
            height: 450,
            fontSize: 54,
            circleThickness: 8,
            colors: {
                circle: '#34495e',
                circleBackground: '#2c3e50',
                text: '#ffffff'
            },
            fontFamily: "Georgia, serif",
            fontWeight: '700',
            labelFontWeight: '400',
            showDays: true,
            showHours: true,
            showMinutes: true,
            showSeconds: true,
            multicolor: {
                enabled: false
            },
            gradientWave: {
                enabled: false,
                colors: ['#667eea', '#764ba2', '#f093fb', '#4facfe'],
                speed: 1.0
            }
        },
        'ocean-breeze': {
            name: 'Ocean Breeze',
            timeType: 'countdown',
            width: 900,
            height: 450,
            fontSize: 56,
            circleThickness: 10,
            colors: {
                circle: '#0ea5e9',
                circleBackground: '#075985',
                text: '#ffffff'
            },
            fontFamily: "'Segoe UI', Tahoma, sans-serif",
            fontWeight: '600',
            labelFontWeight: '400',
            showDays: true,
            showHours: true,
            showMinutes: true,
            showSeconds: true,
            multicolor: {
                enabled: true,
                colors: ['#4facfe', '#00f2fe', '#00d4ff', '#0099ff']
            },
            gradientWave: {
                enabled: false,
                colors: ['#0ea5e9', '#3b82f6', '#60a5fa', '#93c5fd'],
                speed: 1.0
            }
        },
        'sunset-glow': {
            name: 'Sunset Glow',
            timeType: 'countdown',
            width: 900,
            height: 450,
            fontSize: 56,
            circleThickness: 10,
            colors: {
                circle: '#f97316',
                circleBackground: '#c2410c',
                text: '#ffffff'
            },
            fontFamily: "'Segoe UI', Tahoma, sans-serif",
            fontWeight: '700',
            labelFontWeight: '500',
            showDays: true,
            showHours: true,
            showMinutes: true,
            showSeconds: true,
            multicolor: {
                enabled: true,
                colors: ['#f97316', '#fb923c', '#fdba74', '#ff8c42']
            },
            gradientWave: {
                enabled: false,
                colors: ['#f97316', '#fb923c', '#fdba74', '#ff8c42'],
                speed: 1.0
            }
        },
        'forest-green': {
            name: 'Forest Green',
            timeType: 'countdown',
            width: 900,
            height: 450,
            fontSize: 56,
            circleThickness: 10,
            colors: {
                circle: '#22c55e',
                circleBackground: '#166534',
                text: '#ffffff'
            },
            fontFamily: "'Segoe UI', Tahoma, sans-serif",
            fontWeight: '600',
            labelFontWeight: '400',
            showDays: true,
            showHours: true,
            showMinutes: true,
            showSeconds: true,
            multicolor: {
                enabled: true,
                colors: ['#22c55e', '#16a34a', '#15803d', '#10b981']
            },
            gradientWave: {
                enabled: false,
                colors: ['#22c55e', '#16a34a', '#15803d', '#10b981'],
                speed: 1.0
            }
        },
        'black-friday': {
            name: 'Black Friday',
            timeType: 'countdown',
            width: 900,
            height: 450,
            fontSize: 58,
            circleThickness: 12,
            colors: {
                circle: '#dc2626',
                circleBackground: '#991b1b',
                text: '#ffffff'
            },
            fontFamily: "'Impact', 'Arial Black', sans-serif",
            fontWeight: '900',
            labelFontWeight: '700',
            showDays: true,
            showHours: true,
            showMinutes: true,
            showSeconds: true,
            multicolor: {
                enabled: true,
                colors: ['#000000', '#ff0000', '#ff3333', '#cc0000']
            },
            gradientWave: {
                enabled: false,
                colors: ['#dc2626', '#ef4444', '#f87171', '#991b1b'],
                speed: 1.0
            },
            celebration: {
                enabled: true,
                type: 'both',
                intensity: 1.5,
                colors: ['#000000', '#ff0000', '#ffffff', '#ff3333', '#cc0000']
            },
        },
        'christmas': {
            name: 'Christmas',
            timeType: 'countdown',
            width: 900,
            height: 450,
            fontSize: 56,
            circleThickness: 10,
            colors: {
                circle: '#dc2626',
                circleBackground: '#991b1b',
                text: '#ffffff'
            },
            fontFamily: "'Georgia', serif",
            fontWeight: '700',
            labelFontWeight: '500',
            showDays: true,
            showHours: true,
            showMinutes: true,
            showSeconds: true,
            multicolor: {
                enabled: true,
                colors: ['#dc2626', '#16a34a', '#ffffff', '#fbbf24']
            },
            gradientWave: {
                enabled: false,
                colors: ['#dc2626', '#16a34a', '#fbbf24', '#f97316'],
                speed: 1.0
            },
            celebration: {
                enabled: true,
                type: 'confetti',
                intensity: 1.2,
                colors: ['#dc2626', '#16a34a', '#ffffff', '#fbbf24', '#f97316']
            },
        },
        'product-launch': {
            name: 'Product Launch',
            timeType: 'countdown',
            width: 900,
            height: 450,
            fontSize: 56,
            circleThickness: 14,
            colors: {
                circle: '#6366f1',
                circleBackground: '#1e1b4b',
                text: '#ffffff'
            },
            fontFamily: "'Segoe UI', Tahoma, sans-serif",
            fontWeight: '700',
            labelFontWeight: '600',
            showDays: true,
            showHours: true,
            showMinutes: true,
            showSeconds: true,
            multicolor: {
                enabled: true,
                colors: ['#667eea', '#764ba2', '#f093fb', '#4facfe']
            },
            gradientWave: {
                enabled: false,
                colors: ['#6366f1', '#818cf8', '#a78bfa', '#c084fc'],
                speed: 1.0
            },
            celebration: {
                enabled: true,
                type: 'both',
                intensity: 1.3,
                colors: ['#667eea', '#764ba2', '#f093fb', '#4facfe', '#43e97b']
            },
        }
    };
    
    // Current configuration
    let currentConfig = {
        timeType: 'countdown',
        duration: 3600,
        showDays: true,
        showHours: true,
        showMinutes: true,
        showSeconds: true,
        width: 900,
        height: 450,
        fontSize: 56,
        circleThickness: 14,
        colors: {
            circle: '#667eea',
            circleBackground: '#f0f4ff',
            text: '#2c3e50'
        },
        fontFamily: "'Segoe UI', Tahoma, sans-serif",
        fontWeight: '700',
        labelFontWeight: '500',
        multicolor: {
            enabled: false,
            colors: ['#667eea', '#764ba2', '#f093fb', '#4facfe']
        },
        celebration: {
            enabled: true,
            type: 'both',
            intensity: 1.0,
            colors: ['#667eea', '#764ba2', '#f093fb', '#4facfe', '#43e97b', '#fa709a', '#fee140', '#30cfd0']
        },
        pulseGlow: {
            enabled: false,
            intensity: 1.0,
            speed: 1.0
        },
        gradientWave: {
            enabled: false,
            colors: ['#667eea', '#764ba2', '#f093fb', '#4facfe'],
            speed: 1.0
        },
        sound: {
            enabled: false,
            tickEnabled: true,
            tickVolume: 0.3,
            finishEnabled: true,
            finishVolume: 0.5,
            tickFrequency: 800,
            finishFrequency: 600
        },
        effects3D: {
            enabled: false,
            shadowDepth: 20,
            rotationSpeed: 0,
            parallaxIntensity: 0,
            lightingIntensity: 0.5
        },
    };
    
    let timerInstance = null;
    
    // Track current active preset
    let currentActivePreset = 'premium-base';
    
    /**
     * Convert hex color to RGB object
     * @param {string} hex - Hex color code (e.g., '#667eea')
     * @returns {Object|null} RGB object with r, g, b properties or null if invalid
     */
    function hexToRgb(hex) {
        const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
        return result ? {
            r: parseInt(result[1], 16),
            g: parseInt(result[2], 16),
            b: parseInt(result[3], 16)
        } : null;
    }
    
    /**
     * Calculate luminance (brightness) of a color
     * Returns value between 0 (dark) and 1 (light)
     * @param {string} hex - Hex color code
     * @returns {number} Luminance value (0-1)
     */
    function getLuminance(color) {
        // Handle rgba colors
        let hex = color;
        if (color.startsWith('rgba') || color.startsWith('rgb')) {
            const rgbaMatch = color.match(/rgba?\((\d+),\s*(\d+),\s*(\d+)/);
            if (rgbaMatch) {
                const r = parseInt(rgbaMatch[1]);
                const g = parseInt(rgbaMatch[2]);
                const b = parseInt(rgbaMatch[3]);
                hex = '#' + [r, g, b].map(x => {
                    const hex = x.toString(16);
                    return hex.length === 1 ? '0' + hex : hex;
                }).join('');
            } else {
                return 0.5; // Default if can't parse
            }
        }
        
        const rgb = hexToRgb(hex);
        if (!rgb) return 0.5; // Default to medium if invalid
        
        // Calculate relative luminance using WCAG formula
        const r = rgb.r / 255;
        const g = rgb.g / 255;
        const b = rgb.b / 255;
        
        const rLinear = r <= 0.03928 ? r / 12.92 : Math.pow((r + 0.055) / 1.055, 2.4);
        const gLinear = g <= 0.03928 ? g / 12.92 : Math.pow((g + 0.055) / 1.055, 2.4);
        const bLinear = b <= 0.03928 ? b / 12.92 : Math.pow((b + 0.055) / 1.055, 2.4);
        
        return 0.2126 * rLinear + 0.7152 * gLinear + 0.0722 * bLinear;
    }
    
    /**
     * Check if a color is dark (luminance < 0.5)
     * @param {string} hex - Hex color code
     * @returns {boolean} True if color is dark
     */
    function isDarkColor(hex) {
        return getLuminance(hex) < 0.5;
    }
    
    /**
     * Check if a color is very dark (luminance < 0.3)
     * @param {string} hex - Hex color code
     * @returns {boolean} True if color is very dark
     */
    function isVeryDarkColor(hex) {
        return getLuminance(hex) < 0.3;
    }
    
    /**
     * Calculate optimal text color for maximum contrast with background
     * Returns white for dark backgrounds, dark color for light backgrounds
     * @param {string} bgColor - Background color (hex)
     * @param {string} fallbackColor - Fallback color if calculation fails
     * @returns {string} Optimal text color (hex)
     */
    function calculateOptimalTextColor(bgColor, fallbackColor = '#2c3e50') {
        if (!bgColor) return fallbackColor;
        
        // Handle rgba colors - extract base color
        let baseBgColor = bgColor;
        if (bgColor.startsWith('rgba')) {
            // Extract RGB values from rgba
            const rgbaMatch = bgColor.match(/rgba?\((\d+),\s*(\d+),\s*(\d+)/);
            if (rgbaMatch) {
                const r = parseInt(rgbaMatch[1]);
                const g = parseInt(rgbaMatch[2]);
                const b = parseInt(rgbaMatch[3]);
                // Convert to hex
                baseBgColor = '#' + [r, g, b].map(x => {
                    const hex = x.toString(16);
                    return hex.length === 1 ? '0' + hex : hex;
                }).join('');
            }
        }
        
        const bgLuminance = getLuminance(baseBgColor);
        
        // If background is dark (luminance < 0.5), use light text
        if (bgLuminance < 0.5) {
            // Very dark backgrounds get white, moderately dark get light gray
            return bgLuminance < 0.2 ? '#ffffff' : '#e5e7eb';
        } else {
            // Light backgrounds get dark text
            return bgLuminance > 0.9 ? '#1a1a1a' : '#2c3e50';
        }
    }
    
    /**
     * Auto-adjust text color based on background color
     * If background is dark, ensure text is light (white or light color)
     * If background is light, ensure text is dark (but not too dark unless it's black)
     * CRITICAL: Ensures text is ALWAYS readable
     */
    function autoAdjustTextColorForDarkBackground() {
        if (!currentConfig || !currentConfig.colors) return;
        
        const bgColor = currentConfig.colors.circleBackground || '#f0f4ff';
        const currentTextColor = currentConfig.colors.text || '#2c3e50';
        
        // Handle rgba colors - convert to hex for comparison
        let bgColorForComparison = bgColor;
        if (bgColor.startsWith('rgba')) {
            const rgbaMatch = bgColor.match(/rgba?\((\d+),\s*(\d+),\s*(\d+)/);
            if (rgbaMatch) {
                const r = parseInt(rgbaMatch[1]);
                const g = parseInt(rgbaMatch[2]);
                const b = parseInt(rgbaMatch[3]);
                bgColorForComparison = '#' + [r, g, b].map(x => {
                    const hex = x.toString(16);
                    return hex.length === 1 ? '0' + hex : hex;
                }).join('');
            }
        }
        
        // Calculate optimal text color based on background
        const optimalTextColor = calculateOptimalTextColor(bgColor, currentTextColor);
        
        // Check if current text color has good contrast
        let bgLuminance, textLuminance;
        try {
            bgLuminance = getLuminance(bgColorForComparison);
            textLuminance = getLuminance(currentTextColor);
        } catch (e) {
            // If calculation fails, use optimal color
            currentConfig.colors.text = optimalTextColor;
            if ($('#textColor').length) {
                $('#textColor').val(currentConfig.colors.text);
            }
            return;
        }
        
        const contrastRatio = Math.abs(bgLuminance - textLuminance);
        
        // CRITICAL: If contrast is too low (< 0.4) or colors are the same, use optimal color
        const bgColorNormalized = bgColorForComparison.toLowerCase().replace(/\s/g, '').replace('#', '');
        const textColorNormalized = currentTextColor.toLowerCase().replace(/\s/g, '').replace('#', '');
        
        if (contrastRatio < 0.4 || bgColorNormalized === textColorNormalized) {
            // Text and background are too similar or the same - use optimal color
            currentConfig.colors.text = optimalTextColor;
            if ($('#textColor').length) {
                $('#textColor').val(currentConfig.colors.text);
            }
        }
    }
    
    // Initialize
    $(document).ready(function() {
        initializePresets();
        initializeControls();
        // Load default preset to ensure everything is initialized
        loadPreset('premium-base');
        
        // Update controls and preview (preset colors are already corrected in loadPreset)
        updateControlsFromConfig();
        updatePreview();
        updateCodeOutput();
    });
    
    /**
     * Initialize preset buttons in the UI
     * Creates clickable buttons for each preset and attaches event handlers
     */
    function initializePresets() {
        const container = $('#preset-buttons');
        container.empty();
        
        for (const [key, preset] of Object.entries(PRESETS)) {
            const btn = $('<button>')
                .addClass('preset-btn')
                .text(preset.name)
                .attr('data-preset', key)  // Use attr instead of data for better selector compatibility
                .on('click', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    const presetKey = $(this).attr('data-preset');
                    loadPreset(presetKey);
                });
            
            if (key === 'premium-base') {
                btn.addClass('active');
                currentActivePreset = 'premium-base';
            }
            
            container.append(btn);
        }
    }
    
    /**
     * Load a preset configuration
     * @param {string} presetKey - The key of the preset to load
     */
    function loadPreset(presetKey) {
        const preset = PRESETS[presetKey];
        if (!preset) return;
        
        // Update tracked active preset
        currentActivePreset = presetKey;
        
        // Remove active class from all buttons
        $('.preset-btn').removeClass('active');
        
        // Add active class to selected button - try multiple selector methods
        let $activeBtn = $(`.preset-btn[data-preset="${presetKey}"]`);
        if ($activeBtn.length === 0) {
            // Try alternative selector
            $activeBtn = $('.preset-btn').filter(function() {
                return $(this).attr('data-preset') === presetKey;
            });
        }
        if ($activeBtn.length === 0) {
            // Try by text content
            $activeBtn = $('.preset-btn').filter(function() {
                return $(this).text().trim() === PRESETS[presetKey].name;
            });
        }
        
        if ($activeBtn.length) {
            $activeBtn.addClass('active');
        }
        
        // Update current config - ensure all required fields are present
        currentConfig = $.extend(true, {}, preset);
        
        // Ensure required fields have defaults
        if (!currentConfig.timeType) {
            currentConfig.timeType = 'countdown';
        }
        if (!currentConfig.duration) {
            currentConfig.duration = 3600;
        }
        if (typeof currentConfig.showDays === 'undefined') currentConfig.showDays = true;
        if (typeof currentConfig.showHours === 'undefined') currentConfig.showHours = true;
        if (typeof currentConfig.showMinutes === 'undefined') currentConfig.showMinutes = true;
        if (typeof currentConfig.showSeconds === 'undefined') currentConfig.showSeconds = true;
        
        // Ensure pulseGlow has defaults
        if (!currentConfig.pulseGlow) {
            currentConfig.pulseGlow = {
                enabled: false,
                intensity: 1.0,
                speed: 1.0
            };
        }
        
        // Ensure effects3D has defaults
        if (!currentConfig.effects3D) {
            currentConfig.effects3D = {
                enabled: false,
                shadowDepth: 20,
                rotationSpeed: 0,
                parallaxIntensity: 0,
                lightingIntensity: 0.5
            };
        }
        
        // Ensure liquidEffect has defaults (always enabled - distinctive visual feature)
        if (!currentConfig.liquidEffect) {
            currentConfig.liquidEffect = {
                enabled: true,
                intensity: 1.2,
                bubbleSpeed: 1.0,
                waveSpeed: 1.0,
                bubbleSpawnRate: 0.8,
                showEvaporation: true,
                liquidColor: null
            };
        }
        currentConfig.liquidEffect.enabled = true;
        
        // Ensure text color is readable and different from background (only for presets)
        if (currentConfig.colors && currentConfig.colors.circleBackground && currentConfig.colors.text) {
            // Normalize colors for comparison
            let bgColorNormalized = currentConfig.colors.circleBackground.toLowerCase().replace(/\s/g, '').trim();
            let textColorNormalized = currentConfig.colors.text.toLowerCase().replace(/\s/g, '').trim();
            
            // Handle rgba colors - extract hex equivalent
            if (bgColorNormalized.startsWith('rgba') || bgColorNormalized.startsWith('rgb')) {
                const rgbaMatch = bgColorNormalized.match(/rgba?\((\d+),\s*(\d+),\s*(\d+)/);
                if (rgbaMatch) {
                    const r = parseInt(rgbaMatch[1]);
                    const g = parseInt(rgbaMatch[2]);
                    const b = parseInt(rgbaMatch[3]);
                    bgColorNormalized = '#' + [r, g, b].map(x => {
                        const hex = x.toString(16);
                        return hex.length === 1 ? '0' + hex : hex;
                    }).join('');
                }
            }
            
            // Remove # for comparison
            const bgHex = bgColorNormalized.replace('#', '');
            const textHex = textColorNormalized.replace('#', '');
            
            // If colors are identical or too similar, use optimal color (only for presets)
            if (bgHex === textHex || 
                (bgHex.length === textHex.length && bgHex.substring(0, 4) === textHex.substring(0, 4))) {
                // Colors are too similar - use optimal color for preset
                const optimalColor = calculateOptimalTextColor(currentConfig.colors.circleBackground);
                currentConfig.colors.text = optimalColor;
            }
            
            // Additional check: verify contrast ratio is sufficient (only for presets)
            try {
                const bgLuminance = getLuminance(bgColorNormalized);
                const textLuminance = getLuminance(textColorNormalized);
                const contrastRatio = Math.abs(bgLuminance - textLuminance);
                
                // If contrast is too low, force optimal color (only for presets)
                if (contrastRatio < 0.5) {
                    const optimalColor = calculateOptimalTextColor(currentConfig.colors.circleBackground);
                    currentConfig.colors.text = optimalColor;
                }
            } catch (e) {
                // If calculation fails, use optimal color as fallback (only for presets)
                const optimalColor = calculateOptimalTextColor(currentConfig.colors.circleBackground);
                currentConfig.colors.text = optimalColor;
            }
        }
        
        // Update all controls
        updateControlsFromConfig();
        updatePreview();
        updateCodeOutput();
        
        // Maintain active state after DOM updates
        // Use multiple timeouts to ensure state persists after async operations
        const maintainActiveState = function() {
            $('.preset-btn').removeClass('active');
            let $activeBtn = $(`.preset-btn[data-preset="${currentActivePreset}"]`);
            if ($activeBtn.length === 0) {
                $activeBtn = $('.preset-btn').filter(function() {
                    return $(this).attr('data-preset') === currentActivePreset;
                });
            }
            if ($activeBtn.length) {
                $activeBtn.addClass('active');
            }
        };
        
        setTimeout(maintainActiveState, 10);
        setTimeout(maintainActiveState, 100);
        setTimeout(maintainActiveState, 500);
    }
    
    /**
     * Initialize all control event handlers
     * Attaches event listeners to all form controls in the customizer panel
     */
    function initializeControls() {
        // Timer Type
        $('#timeType').on('change', function() {
            currentConfig.timeType = $(this).val();
            updatePreview();
            updateCodeOutput();
        });
        
        // Duration
        $('#duration').on('input', function() {
            currentConfig.duration = parseInt($(this).val()) || 3600;
            updatePreview();
            updateCodeOutput();
        });
        
        // Show Units
        $('#showDays, #showHours, #showMinutes, #showSeconds').on('change', function() {
            const id = $(this).attr('id');
            const key = id.replace('show', '').toLowerCase();
            currentConfig['show' + key.charAt(0).toUpperCase() + key.slice(1)] = $(this).is(':checked');
            updatePreview();
            updateCodeOutput();
        });
        
        // Size controls
        $('#width').on('input', function() {
            const val = parseInt($(this).val());
            currentConfig.width = val;
            $('#widthValue').text(val);
            updatePreview();
            updateCodeOutput();
        });
        
        $('#height').on('input', function() {
            const val = parseInt($(this).val());
            currentConfig.height = val;
            $('#heightValue').text(val);
            updatePreview();
            updateCodeOutput();
        });
        
        $('#fontSize').on('input', function() {
            const val = parseInt($(this).val());
            currentConfig.fontSize = val;
            $('#fontSizeValue').text(val);
            updatePreview();
            updateCodeOutput();
        });
        
        $('#circleThickness').on('input', function() {
            const val = parseInt($(this).val());
            currentConfig.circleThickness = val;
            $('#circleThicknessValue').text(val);
            updatePreview();
            updateCodeOutput();
        });
        
        // Color controls - ensure they are always functional
        $('#circleColor').on('input change', function() {
            const colorValue = $(this).val();
            if (colorValue && currentConfig && currentConfig.colors) {
                currentConfig.colors.circle = colorValue;
                updatePreview();
                updateCodeOutput();
            }
        });
        
        $('#circleBackgroundColor').on('input change', function() {
            const colorValue = $(this).val();
            if (colorValue && currentConfig && currentConfig.colors) {
                currentConfig.colors.circleBackground = colorValue;
                // User is free to modify colors as they wish - no restrictions
                updatePreview();
                updateCodeOutput();
            }
        });
        
        $('#textColor').on('input', function() {
            currentConfig.colors.text = $(this).val();
            // User is free to modify colors as they wish - no restrictions
            updatePreview();
            updateCodeOutput();
        });
        
        // Font controls
        $('#fontFamily').on('change', function() {
            currentConfig.fontFamily = $(this).val();
            updatePreview();
            updateCodeOutput();
        });
        
        $('#fontWeight').on('change', function() {
            currentConfig.fontWeight = $(this).val();
            updatePreview();
            updateCodeOutput();
        });
        
        $('#labelFontWeight').on('change', function() {
            currentConfig.labelFontWeight = $(this).val();
            updatePreview();
            updateCodeOutput();
        });
        
        // Multicolor
        $('#multicolorEnabled').on('change', function() {
            const isEnabled = $(this).is(':checked');
            currentConfig.multicolor.enabled = isEnabled;
            if (isEnabled) {
                $('#multicolorColors').show();
                // Ensure we have valid colors
                if (!currentConfig.multicolor.colors || currentConfig.multicolor.colors.length === 0) {
                    currentConfig.multicolor.colors = ['#667eea', '#764ba2', '#f093fb', '#4facfe'];
                }
            } else {
                $('#multicolorColors').hide();
            }
            updatePreview();
            updateCodeOutput();
        });
        
        $('#multicolorColorsInput').on('input', function() {
            const inputValue = $(this).val();
            // Parse colors and validate hex format
            const colors = inputValue.split(',')
                .map(c => c.trim())
                .filter(c => {
                    // Validate hex color format
                    return /^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/.test(c);
                });
            
            // Only update if we have valid colors
            if (colors.length > 0) {
                currentConfig.multicolor.colors = colors;
                updatePreview();
                updateCodeOutput();
            } else if (inputValue.trim() === '') {
                // If input is empty, use default
                currentConfig.multicolor.colors = ['#667eea'];
                updatePreview();
                updateCodeOutput();
            }
        });
        
        // Pulse/Glow controls
        $('#pulseGlowEnabled').on('change', function() {
            const isEnabled = $(this).is(':checked');
            currentConfig.pulseGlow.enabled = isEnabled;
            if (isEnabled) {
                $('#pulseGlowControls, #pulseGlowSpeedContainer').show();
            } else {
                $('#pulseGlowControls, #pulseGlowSpeedContainer').hide();
            }
            updatePreview();
            updateCodeOutput();
        });
        
        $('#pulseGlowIntensity').on('input', function() {
            const val = parseFloat($(this).val()) || 1.0;
            // Ensure pulseGlow object exists
            if (!currentConfig.pulseGlow) {
                currentConfig.pulseGlow = {
                    enabled: false,
                    intensity: 1.0,
                    speed: 1.0
                };
            }
            currentConfig.pulseGlow.intensity = val;
            $('#pulseGlowIntensityValue').text(val.toFixed(1));
            updatePreview();
            updateCodeOutput();
        });
        
        $('#pulseGlowSpeed').on('input', function() {
            const val = parseFloat($(this).val()) || 1.0;
            // Ensure pulseGlow object exists
            if (!currentConfig.pulseGlow) {
                currentConfig.pulseGlow = {
                    enabled: false,
                    intensity: 1.0,
                    speed: 1.0
                };
            }
            currentConfig.pulseGlow.speed = val;
            $('#pulseGlowSpeedValue').text(val.toFixed(1));
            updatePreview();
            updateCodeOutput();
        });
        
        // Gradient Wave controls
        $('#gradientWaveEnabled').on('change', function() {
            const isEnabled = $(this).is(':checked');
            // Ensure gradientWave object exists
            if (!currentConfig.gradientWave) {
                currentConfig.gradientWave = {
                    enabled: false,
                    colors: ['#667eea', '#764ba2', '#f093fb', '#4facfe'],
                    speed: 1.0
                };
            }
            currentConfig.gradientWave.enabled = isEnabled;
            if (isEnabled) {
                $('#gradientWaveControls, #gradientWaveSpeedContainer').show();
                // Disable multicolor if gradient wave is enabled
                if (currentConfig.multicolor && currentConfig.multicolor.enabled) {
                    currentConfig.multicolor.enabled = false;
                    $('#multicolorEnabled').prop('checked', false);
                    $('#multicolorColors').hide();
                }
            } else {
                $('#gradientWaveControls, #gradientWaveSpeedContainer').hide();
            }
            updatePreview();
            updateCodeOutput();
        });
        
        $('#gradientWaveColorsInput').on('input', function() {
            const colorsStr = $(this).val();
            const colors = colorsStr.split(',').map(c => c.trim()).filter(c => /^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/.test(c));
            if (colors.length >= 2) {
                currentConfig.gradientWave.colors = colors;
            } else {
                currentConfig.gradientWave.colors = ['#667eea', '#764ba2', '#f093fb', '#4facfe'];
            }
            updatePreview();
            updateCodeOutput();
        });
        
        $('#gradientWaveSpeed').on('input', function() {
            const val = parseFloat($(this).val()) || 1.0;
            // Ensure gradientWave object exists
            if (!currentConfig.gradientWave) {
                currentConfig.gradientWave = {
                    enabled: false,
                    colors: ['#667eea', '#764ba2', '#f093fb', '#4facfe'],
                    speed: 1.0
                };
            }
            currentConfig.gradientWave.speed = val;
            $('#gradientWaveSpeedValue').text(val.toFixed(1));
            updatePreview();
            updateCodeOutput();
        });
        
        // Disable gradient wave if multicolor is enabled
        $('#multicolorEnabled').on('change', function() {
            if ($(this).is(':checked') && currentConfig.gradientWave && currentConfig.gradientWave.enabled) {
                currentConfig.gradientWave.enabled = false;
                $('#gradientWaveEnabled').prop('checked', false);
                $('#gradientWaveControls, #gradientWaveSpeedContainer').hide();
            }
        });
        
        // Celebration controls
        $('#celebrationEnabled').on('change', function() {
            const isEnabled = $(this).is(':checked');
            // Ensure celebration object exists
            if (!currentConfig.celebration) {
                currentConfig.celebration = {
                    enabled: true,
                    type: 'both',
                    intensity: 1.0,
                    colors: ['#667eea', '#764ba2', '#f093fb', '#4facfe', '#43e97b', '#fa709a', '#fee140', '#30cfd0']
                };
            }
            currentConfig.celebration.enabled = isEnabled;
            if (isEnabled) {
                $('#celebrationControls, #celebrationIntensityContainer').show();
            } else {
                $('#celebrationControls, #celebrationIntensityContainer').hide();
            }
            updatePreview();
            updateCodeOutput();
        });
        
        $('#celebrationType').on('change', function() {
            // Ensure celebration object exists
            if (!currentConfig.celebration) {
                currentConfig.celebration = {
                    enabled: true,
                    type: 'both',
                    intensity: 1.0,
                    colors: ['#667eea', '#764ba2', '#f093fb', '#4facfe', '#43e97b', '#fa709a', '#fee140', '#30cfd0']
                };
            }
            currentConfig.celebration.type = $(this).val();
            updatePreview();
            updateCodeOutput();
        });
        
        $('#celebrationIntensity').on('input', function() {
            const val = parseFloat($(this).val()) || 1.0;
            // Ensure celebration object exists
            if (!currentConfig.celebration) {
                currentConfig.celebration = {
                    enabled: true,
                    type: 'both',
                    intensity: 1.0,
                    colors: ['#667eea', '#764ba2', '#f093fb', '#4facfe', '#43e97b', '#fa709a', '#fee140', '#30cfd0']
                };
            }
            currentConfig.celebration.intensity = val;
            $('#celebrationIntensityValue').text(val.toFixed(1));
            updatePreview();
            updateCodeOutput();
        });
        
        // Test Celebration button - trigger animation immediately
        $('#testCelebrationBtn').on('click', function() {
            const $preview = $('#timer-preview');
            const timerInstance = $preview.data('countdownTimerPro');
            
            if (timerInstance && typeof timerInstance.startCelebration === 'function') {
                // Stop current celebration if running
                if (typeof timerInstance.stopCelebration === 'function') {
                    timerInstance.stopCelebration();
                }
                
                // Update celebration config with current settings
                if (!timerInstance.options.celebration) {
                    timerInstance.options.celebration = {};
                }
                timerInstance.options.celebration.enabled = currentConfig.celebration.enabled;
                timerInstance.options.celebration.type = currentConfig.celebration.type || 'both';
                timerInstance.options.celebration.intensity = currentConfig.celebration.intensity || 1.0;
                timerInstance.options.celebration.colors = currentConfig.celebration.colors || ['#667eea', '#764ba2', '#f093fb', '#4facfe', '#43e97b', '#fa709a', '#fee140', '#30cfd0'];
                
                // Start celebration animation
                timerInstance.startCelebration();
            }
        });
        
        // Initialize celebration controls visibility
        if (currentConfig.celebration && currentConfig.celebration.enabled) {
            $('#celebrationControls, #celebrationIntensityContainer').show();
        } else {
            $('#celebrationControls, #celebrationIntensityContainer').hide();
        }
        
        // Sound Effects controls
        $('#soundEnabled').on('change', function() {
            const isEnabled = $(this).is(':checked');
            currentConfig.sound.enabled = isEnabled;
            if (isEnabled) {
                $('#soundControls').show();
            } else {
                $('#soundControls').hide();
            }
            updatePreview();
            updateCodeOutput();
        });
        
        $('#soundTickEnabled').on('change', function() {
            currentConfig.sound.tickEnabled = $(this).is(':checked');
            updatePreview();
            updateCodeOutput();
        });
        
        $('#soundFinishEnabled').on('change', function() {
            currentConfig.sound.finishEnabled = $(this).is(':checked');
            updatePreview();
            updateCodeOutput();
        });
        
        $('#soundTickVolume').on('input', function() {
            const val = parseFloat($(this).val());
            currentConfig.sound.tickVolume = val;
            $('#soundTickVolumeValue').text(val.toFixed(1));
            updatePreview();
            updateCodeOutput();
        });
        
        // 3D Effects controls
        $('#effects3DEnabled').on('change', function() {
            const isEnabled = $(this).is(':checked');
            // Ensure effects3D object exists
            if (!currentConfig.effects3D) {
                currentConfig.effects3D = {
                    enabled: false,
                    shadowDepth: 20,
                    rotationSpeed: 0,
                    parallaxIntensity: 0,
                    lightingIntensity: 0.5
                };
            }
            currentConfig.effects3D.enabled = isEnabled;
            if (isEnabled) {
                $('#effects3DControls, #effects3DParallaxContainer, #effects3DRotationContainer').show();
            } else {
                $('#effects3DControls, #effects3DParallaxContainer, #effects3DRotationContainer').hide();
            }
            updatePreview();
            updateCodeOutput();
        });
        
        $('#effects3DShadowDepth').on('input', function() {
            const val = parseInt($(this).val());
            currentConfig.effects3D.shadowDepth = val;
            $('#effects3DShadowDepthValue').text(val);
            updatePreview();
            updateCodeOutput();
        });
        
        $('#effects3DParallax').on('input', function() {
            const val = parseFloat($(this).val()) || 0;
            // Ensure effects3D object exists
            if (!currentConfig.effects3D) {
                currentConfig.effects3D = {
                    enabled: false,
                    shadowDepth: 20,
                    rotationSpeed: 0,
                    parallaxIntensity: 0,
                    lightingIntensity: 0.5
                };
            }
            currentConfig.effects3D.parallaxIntensity = val;
            $('#effects3DParallaxValue').text(val.toFixed(1));
            updatePreview();
            updateCodeOutput();
        });
        
        $('#effects3DRotation').on('input', function() {
            const val = parseFloat($(this).val()) || 0;
            // Ensure effects3D object exists
            if (!currentConfig.effects3D) {
                currentConfig.effects3D = {
                    enabled: false,
                    shadowDepth: 20,
                    rotationSpeed: 0,
                    parallaxIntensity: 0,
                    lightingIntensity: 0.5
                };
            }
            currentConfig.effects3D.rotationSpeed = val;
            $('#effects3DRotationValue').text(val.toFixed(2));
            updatePreview();
            updateCodeOutput();
        });
        
        // Cross-Tab Synchronization controls (FEATURE 1)
        $('#realTimeSyncEnabled').on('change', function() {
            const isEnabled = $(this).is(':checked');
            // Ensure realTimeSync object exists
            if (!currentConfig.realTimeSync) {
                currentConfig.realTimeSync = {
                    enabled: true,
                    roomId: 'demo-room-xtremecode',
                    serverUrl: null,
                    showViewerCount: true,
                    showConnectionStatus: false,
                    syncInterval: 1000,
                    reconnectInterval: 3000,
                    maxReconnectAttempts: 5,
                    useLocalStorageFallback: true
                };
            }
            currentConfig.realTimeSync.enabled = isEnabled;
            if (isEnabled) {
                $('#realTimeSyncControls').show();
            } else {
                $('#realTimeSyncControls').hide();
            }
            updatePreview();
            updateCodeOutput();
        });
        
        $('#showViewerCount').on('change', function() {
            const isEnabled = $(this).is(':checked');
            // Ensure realTimeSync object exists
            if (!currentConfig.realTimeSync) {
                currentConfig.realTimeSync = {
                    enabled: true,
                    roomId: 'demo-room-xtremecode',
                    serverUrl: null,
                    showViewerCount: true,
                    showConnectionStatus: false,
                    syncInterval: 1000,
                    reconnectInterval: 3000,
                    maxReconnectAttempts: 5,
                    useLocalStorageFallback: true
                };
            }
            currentConfig.realTimeSync.showViewerCount = isEnabled;
            updatePreview();
            updateCodeOutput();
        });
        
        // Behavioral Targeting controls (FEATURE 2)
        $('#behavioralTargetingEnabled').on('change', function() {
            const isEnabled = $(this).is(':checked');
            // Ensure behavioralTargeting object exists
            if (!currentConfig.behavioralTargeting) {
                currentConfig.behavioralTargeting = {
                    enabled: false,
                    rules: [
                        {
                            condition: 'newVisitor',
                            duration: 86400, // 24 hours
                            message: 'Offerta valida per 24 ore!'
                        },
                        {
                            condition: 'returningVisitor',
                            duration: 7200, // 2 hours
                            message: 'Ultime 2 ore! Non perdere l\'occasione!'
                        },
                        {
                            condition: 'mobile',
                            duration: 3600, // 1 hour
                            message: 'Solo 1 ora rimasta!'
                        },
                        {
                            condition: 'desktop',
                            duration: 86400, // 24 hours
                            message: 'Offerta valida per 24 ore'
                        },
                        {
                            condition: 'socialMedia',
                            duration: 3600, // 1 hour
                            message: 'Offerta limitata! Solo 1 ora!'
                        },
                        {
                            condition: 'email',
                            duration: 21600, // 6 hours
                            message: 'Offerta valida per 6 ore'
                        },
                        {
                            condition: 'direct',
                            duration: 86400, // 24 hours
                            message: 'Offerta valida per 24 ore'
                        }
                    ],
                    priority: ['newVisitor', 'returningVisitor', 'device', 'source'],
                    storageKey: 'countdown_timer_visits',
                    visitThreshold: 2
                };
            }
            currentConfig.behavioralTargeting.enabled = isEnabled;
            if (isEnabled) {
                $('#behavioralTargetingControls').show();
            } else {
                $('#behavioralTargetingControls').hide();
            }
            updatePreview();
            updateCodeOutput();
        });
        
        // Conversion Optimization controls (FEATURE 3)
        $('#aiOptimizationEnabled').on('change', function() {
            const isEnabled = $(this).is(':checked');
            // Ensure aiOptimization object exists
            if (!currentConfig.aiOptimization) {
                currentConfig.aiOptimization = {
                    enabled: false,
                    trackInteractions: true,
                    trackConversions: true,
                    learningRate: 0.1,
                    minSamples: 10,
                    optimizationInterval: 3600000, // 1 hour
                    storageKey: 'countdown_timer_ai_data'
                };
            }
            currentConfig.aiOptimization.enabled = isEnabled;
            if (isEnabled) {
                $('#aiOptimizationControls').show();
            } else {
                $('#aiOptimizationControls').hide();
            }
            updatePreview();
            updateCodeOutput();
        });
        
        $('#trackInteractions').on('change', function() {
            const isEnabled = $(this).is(':checked');
            if (!currentConfig.aiOptimization) {
                currentConfig.aiOptimization = {
                    enabled: true,
                    trackInteractions: true,
                    trackConversions: true,
                    learningRate: 0.1,
                    minSamples: 10,
                    optimizationInterval: 3600000,
                    storageKey: 'countdown_timer_ai_data'
                };
            }
            currentConfig.aiOptimization.trackInteractions = isEnabled;
            updatePreview();
            updateCodeOutput();
        });
        
        $('#trackConversions').on('change', function() {
            const isEnabled = $(this).is(':checked');
            if (!currentConfig.aiOptimization) {
                currentConfig.aiOptimization = {
                    enabled: true,
                    trackInteractions: true,
                    trackConversions: true,
                    learningRate: 0.1,
                    minSamples: 10,
                    optimizationInterval: 3600000,
                    storageKey: 'countdown_timer_ai_data'
                };
            }
            currentConfig.aiOptimization.trackConversions = isEnabled;
            updatePreview();
            updateCodeOutput();
        });
        
        // Copy button
        $('#copyBtn').on('click', function() {
            copyCodeToClipboard();
        });
        
        // Export Preset
        $('#exportPresetBtn').on('click', function() {
            exportPreset();
        });
        
        // Import Preset
        $('#importPresetBtn').on('click', function() {
            $('#importPresetFile').click();
        });
        
        $('#importPresetFile').on('change', function(e) {
            const file = e.target.files[0];
            if (file) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    try {
                        const config = JSON.parse(e.target.result);
                        loadPresetFromConfig(config);
                        $('#importPresetBtn').text('✓ Imported').css('background', '#28a745');
                        setTimeout(() => {
                            $('#importPresetBtn').text('📂 Import').css('background', '#17a2b8');
                        }, 2000);
                    } catch (err) {
                        alert('Error: Invalid preset file format.');
                    }
                };
                reader.readAsText(file);
            }
        });
        
        // Share via URL
        $('#shareUrlBtn').on('click', function() {
            shareViaUrl();
        });
        
        // Responsive Preview Toggle
        $('#responsiveToggle').on('click', function() {
            $('#responsivePreview').toggle();
            $(this).text($('#responsivePreview').is(':visible') ? '✕ Close' : '📱 Responsive');
        });
        
        // Responsive Preview Buttons
        $(document).on('click', '.responsive-btn', function() {
            const width = $(this).data('width');
            $('.responsive-btn').removeClass('active').css({
                'background': '#fff',
                'color': 'inherit'
            });
            $(this).addClass('active').css({
                'background': '#667eea',
                'color': '#fff'
            });
            
            const $container = $('.timer-preview-container');
            const $preview = $('#timer-preview');
            
            if (width === 'auto') {
                $container.css({
                    'max-width': '100%',
                    'width': 'auto',
                    'height': 'auto',
                    'min-height': '450px'
                });
                $preview.css({
                    'width': '100%',
                    'height': 'auto'
                });
            } else {
                const widthNum = parseInt(width);
                // Calculate proportional height based on width
                let height;
                if (widthNum <= 400) {
                    // Mobile - ensure sufficient height for vertical layout
                    height = Math.max(700, widthNum * 2.8);
                } else if (widthNum <= 768) {
                    // Tablet - maintain aspect ratio
                    height = Math.max(450, Math.round(widthNum * 0.5));
                } else if (widthNum <= 1024) {
                    // Desktop - maintain aspect ratio
                    height = Math.max(500, Math.round(widthNum * 0.55));
                } else {
                    // Large screens - use more height for better visibility
                    height = Math.max(600, Math.min(800, Math.round(widthNum * 0.45)));
                }
                // For large screens, scale down to fit container if needed
                const containerMaxWidth = $container.parent().width() || 1000;
                let finalWidth = widthNum;
                let finalHeight = height;
                
                if (widthNum > containerMaxWidth - 40) {
                    // Scale down proportionally to fit container (with 40px padding)
                    const scale = (containerMaxWidth - 40) / widthNum;
                    finalWidth = Math.floor(widthNum * scale);
                    finalHeight = Math.floor(height * scale);
                }
                
                $container.css({
                    'max-width': finalWidth + 'px',
                    'width': finalWidth + 'px',
                    'height': finalHeight + 'px',
                    'min-height': finalHeight + 'px',
                    'overflow': 'auto'
                });
                // Also set the timer element size explicitly
                $preview.css({
                    'width': finalWidth + 'px',
                    'height': finalHeight + 'px',
                    'min-width': finalWidth + 'px',
                    'min-height': finalHeight + 'px'
                });
                
                // Update timer options to match responsive width (CRITICAL for mobile vertical layout)
                if (typeof currentConfig !== 'undefined' && currentConfig) {
                    currentConfig.width = finalWidth;
                    currentConfig.height = finalHeight;
                }
                
                // CRITICAL: Force timer instance update BEFORE updatePreview for mobile
                const timerInstance = $preview.data('countdownTimerPro');
                if (timerInstance && finalWidth < 600) {
                    // Immediately set options to force mobile detection
                    timerInstance.options.width = finalWidth;
                    timerInstance.options.height = finalHeight;
                }
            }
            
            // Update preview after resize with delay to ensure DOM is updated
            setTimeout(function() {
                if (typeof updatePreview === 'function') {
                    updatePreview();
                }
                
                // After updatePreview, get the new timer instance and force resize/draw for mobile
                setTimeout(function() {
                    const timerInstance = $preview.data('countdownTimerPro');
                    if (timerInstance) {
                        // If mobile width (< 600), force vertical layout
                        const activeBtn = $('.responsive-btn.active');
                        if (activeBtn.length) {
                            const widthData = activeBtn.data('width');
                            if (widthData && widthData !== 'auto') {
                                const widthNum = parseInt(widthData);
                                if (widthNum < 600) {
                                    // CRITICAL: Force mobile layout by setting width < 600
                                    timerInstance.options.width = widthNum;
                                    timerInstance.options.height = Math.max(700, widthNum * 2.8);
                                    
                                    // Force multiple resize/draw cycles to ensure vertical layout is applied
                                    setTimeout(function() {
                                        if (typeof timerInstance.resize === 'function') {
                                            timerInstance.resize();
                                        }
                                        if (typeof timerInstance.draw === 'function') {
                                            timerInstance.draw();
                                        }
                                    }, 50);
                                    
                                    setTimeout(function() {
                                        if (typeof timerInstance.resize === 'function') {
                                            timerInstance.resize();
                                        }
                                        if (typeof timerInstance.draw === 'function') {
                                            timerInstance.draw();
                                        }
                                    }, 150);
                                }
                            }
                        }
                    }
                }, 100);
            }, 200);
        });
        
        // Export Image
        $('#exportImageBtn').on('click', function() {
            exportAsImage();
        });
        
        // Initialize default values display
        $('#widthValue').text(currentConfig.width);
        $('#heightValue').text(currentConfig.height);
        $('#fontSizeValue').text(currentConfig.fontSize);
        $('#circleThicknessValue').text(currentConfig.circleThickness);
    }
    
    /**
     * Update controls from current config
     */
    /**
     * Update all control values from current configuration
     * Synchronizes UI controls with the currentConfig object
     */
    function updateControlsFromConfig() {
        // Ensure timeType has a default value
        if (!currentConfig.timeType) {
            currentConfig.timeType = 'countdown';
        }
        $('#timeType').val(currentConfig.timeType || 'countdown');
        
        // Ensure duration has a default value
        if (!currentConfig.duration) {
            currentConfig.duration = 3600;
        }
        $('#duration').val(currentConfig.duration);
        
        // Ensure show flags have default values
        if (typeof currentConfig.showDays === 'undefined') currentConfig.showDays = true;
        if (typeof currentConfig.showHours === 'undefined') currentConfig.showHours = true;
        if (typeof currentConfig.showMinutes === 'undefined') currentConfig.showMinutes = true;
        if (typeof currentConfig.showSeconds === 'undefined') currentConfig.showSeconds = true;
        
        $('#showDays').prop('checked', currentConfig.showDays);
        $('#showHours').prop('checked', currentConfig.showHours);
        $('#showMinutes').prop('checked', currentConfig.showMinutes);
        $('#showSeconds').prop('checked', currentConfig.showSeconds);
        
        $('#width').val(currentConfig.width);
        $('#widthValue').text(currentConfig.width);
        $('#height').val(currentConfig.height);
        $('#heightValue').text(currentConfig.height);
        $('#fontSize').val(currentConfig.fontSize);
        $('#fontSizeValue').text(currentConfig.fontSize);
        $('#circleThickness').val(currentConfig.circleThickness);
        $('#circleThicknessValue').text(currentConfig.circleThickness);
        
        // Update color controls - handle rgba colors by converting to hex
        let circleColor = currentConfig.colors.circle || '#667eea';
        let circleBgColor = currentConfig.colors.circleBackground || '#f0f4ff';
        let textColor = currentConfig.colors.text || '#ffffff';
        
        // Convert rgba to hex if needed (input type="color" doesn't support rgba)
        if (circleColor.startsWith('rgba') || circleColor.startsWith('rgb')) {
            const rgbaMatch = circleColor.match(/rgba?\((\d+),\s*(\d+),\s*(\d+)/);
            if (rgbaMatch) {
                const r = parseInt(rgbaMatch[1]);
                const g = parseInt(rgbaMatch[2]);
                const b = parseInt(rgbaMatch[3]);
                circleColor = '#' + [r, g, b].map(x => {
                    const hex = x.toString(16);
                    return hex.length === 1 ? '0' + hex : hex;
                }).join('');
            }
        }
        
        if (circleBgColor.startsWith('rgba') || circleBgColor.startsWith('rgb')) {
            const rgbaMatch = circleBgColor.match(/rgba?\((\d+),\s*(\d+),\s*(\d+)/);
            if (rgbaMatch) {
                const r = parseInt(rgbaMatch[1]);
                const g = parseInt(rgbaMatch[2]);
                const b = parseInt(rgbaMatch[3]);
                circleBgColor = '#' + [r, g, b].map(x => {
                    const hex = x.toString(16);
                    return hex.length === 1 ? '0' + hex : hex;
                }).join('');
            }
        }
        
        if (textColor.startsWith('rgba') || textColor.startsWith('rgb')) {
            const rgbaMatch = textColor.match(/rgba?\((\d+),\s*(\d+),\s*(\d+)/);
            if (rgbaMatch) {
                const r = parseInt(rgbaMatch[1]);
                const g = parseInt(rgbaMatch[2]);
                const b = parseInt(rgbaMatch[3]);
                textColor = '#' + [r, g, b].map(x => {
                    const hex = x.toString(16);
                    return hex.length === 1 ? '0' + hex : hex;
                }).join('');
            }
        }
        
        // Update color inputs - ensure elements exist and values are set correctly
        const $circleColorInput = $('#circleColor');
        const $circleBgColorInput = $('#circleBackgroundColor');
        const $textColorInput = $('#textColor');
        
        if ($circleColorInput.length) {
            $circleColorInput.val(circleColor);
            // Force update using native DOM property
            if ($circleColorInput[0]) {
                $circleColorInput[0].value = circleColor;
            }
        }
        
        if ($circleBgColorInput.length) {
            $circleBgColorInput.val(circleBgColor);
            // Force update using native DOM property
            if ($circleBgColorInput[0]) {
                $circleBgColorInput[0].value = circleBgColor;
            }
        }
        
        if ($textColorInput.length) {
            $textColorInput.val(textColor);
            // Force update using native DOM property
            if ($textColorInput[0]) {
                $textColorInput[0].value = textColor;
            }
        }
        
        $('#fontFamily').val(currentConfig.fontFamily);
        $('#fontWeight').val(currentConfig.fontWeight);
        $('#labelFontWeight').val(currentConfig.labelFontWeight);
        
        $('#multicolorEnabled').prop('checked', currentConfig.multicolor.enabled);
        if (currentConfig.multicolor.enabled) {
            // Ensure colors array is valid
            if (!currentConfig.multicolor.colors || currentConfig.multicolor.colors.length === 0) {
                currentConfig.multicolor.colors = ['#667eea', '#764ba2', '#f093fb', '#4facfe'];
            }
            // Validate all colors are valid hex
            currentConfig.multicolor.colors = currentConfig.multicolor.colors.filter(c => {
                return /^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/.test(c);
            });
            if (currentConfig.multicolor.colors.length === 0) {
                currentConfig.multicolor.colors = ['#667eea'];
            }
            $('#multicolorColors').show();
            $('#multicolorColorsInput').val(currentConfig.multicolor.colors.join(', '));
        } else {
            $('#multicolorColors').hide();
        }
        
        // Celebration controls
        if (!currentConfig.celebration) {
            currentConfig.celebration = {
                enabled: true,
                type: 'both',
                intensity: 1.0
            };
        }
        $('#celebrationEnabled').prop('checked', currentConfig.celebration.enabled);
        $('#celebrationType').val(currentConfig.celebration.type || 'both');
        const celebrationIntensity = currentConfig.celebration.intensity || 1.0;
        $('#celebrationIntensity').val(celebrationIntensity).trigger('input');
        $('#celebrationIntensityValue').text(celebrationIntensity.toFixed(1));
        if (currentConfig.celebration.enabled) {
            $('#celebrationControls, #celebrationIntensityContainer').show();
        } else {
            $('#celebrationControls, #celebrationIntensityContainer').hide();
        }
        
        // Pulse/Glow controls
        if (!currentConfig.pulseGlow) {
            currentConfig.pulseGlow = {
                enabled: false,
                intensity: 1.0,
                speed: 1.0
            };
        }
        $('#pulseGlowEnabled').prop('checked', currentConfig.pulseGlow.enabled);
        const pulseIntensity = currentConfig.pulseGlow.intensity || 1.0;
        $('#pulseGlowIntensity').val(pulseIntensity).trigger('input');
        $('#pulseGlowIntensityValue').text(pulseIntensity.toFixed(1));
        const pulseSpeed = currentConfig.pulseGlow.speed || 1.0;
        $('#pulseGlowSpeed').val(pulseSpeed).trigger('input');
        $('#pulseGlowSpeedValue').text(pulseSpeed.toFixed(1));
        if (currentConfig.pulseGlow.enabled) {
            $('#pulseGlowControls, #pulseGlowSpeedContainer').show();
        } else {
            $('#pulseGlowControls, #pulseGlowSpeedContainer').hide();
        }
        
        // Gradient Wave controls
        if (!currentConfig.gradientWave) {
            currentConfig.gradientWave = {
                enabled: false,
                colors: ['#667eea', '#764ba2', '#f093fb', '#4facfe'],
                speed: 1.0
            };
        }
        $('#gradientWaveEnabled').prop('checked', currentConfig.gradientWave.enabled);
        $('#gradientWaveColorsInput').val((currentConfig.gradientWave.colors || ['#667eea', '#764ba2', '#f093fb', '#4facfe']).join(', '));
        const gradientWaveSpeed = currentConfig.gradientWave.speed || 1.0;
        $('#gradientWaveSpeed').val(gradientWaveSpeed).trigger('input');
        $('#gradientWaveSpeedValue').text(gradientWaveSpeed.toFixed(1));
            if (currentConfig.gradientWave.enabled) {
                $('#gradientWaveControls, #gradientWaveSpeedContainer').show();
            } else {
                $('#gradientWaveControls, #gradientWaveSpeedContainer').hide();
            }
        
        // Sound controls
        if (!currentConfig.sound) {
            currentConfig.sound = {
                enabled: false,
                tickEnabled: false,
                tickVolume: 0.3,
                finishEnabled: true,
                finishVolume: 0.5,
                tickFrequency: 800,
                finishFrequency: 600
            };
        }
        $('#soundEnabled').prop('checked', currentConfig.sound.enabled);
        $('#soundTickEnabled').prop('checked', currentConfig.sound.tickEnabled);
        $('#soundFinishEnabled').prop('checked', currentConfig.sound.finishEnabled);
        $('#soundTickVolume').val(currentConfig.sound.tickVolume || 0.3);
        $('#soundTickVolumeValue').text((currentConfig.sound.tickVolume || 0.3).toFixed(1));
        if (currentConfig.sound.enabled) {
            $('#soundControls').show();
        } else {
            $('#soundControls').hide();
        }
        
        // 3D Effects controls
        if (!currentConfig.effects3D) {
            currentConfig.effects3D = {
                enabled: false,
                shadowDepth: 20,
                rotationSpeed: 0,
                parallaxIntensity: 0,
                lightingIntensity: 0.5
            };
        }
        $('#effects3DEnabled').prop('checked', currentConfig.effects3D.enabled);
        $('#effects3DShadowDepth').val(currentConfig.effects3D.shadowDepth || 20);
        $('#effects3DShadowDepthValue').text(currentConfig.effects3D.shadowDepth || 20);
        const parallaxVal = currentConfig.effects3D.parallaxIntensity || 0;
        $('#effects3DParallax').val(parallaxVal).trigger('input');
        $('#effects3DParallaxValue').text(parallaxVal.toFixed(1));
        const rotationVal = currentConfig.effects3D.rotationSpeed || 0;
        $('#effects3DRotation').val(rotationVal).trigger('input');
        
        // Liquid Effect (always enabled by default - distinctive visual feature, high definition)
        if (!currentConfig.liquidEffect) {
            currentConfig.liquidEffect = {
                enabled: true, // Always enabled - distinctive visual feature
                intensity: 1.2, // High definition - more bubbles and smoother waves
                bubbleSpeed: 1.0,
                waveSpeed: 1.0,
                bubbleSpawnRate: 0.8, // Higher rate for more realistic effect
                showEvaporation: true,
                liquidColor: null
            };
        }
        // Force enabled - always active
        currentConfig.liquidEffect.enabled = true;
        $('#effects3DRotationValue').text(rotationVal.toFixed(2));
        if (currentConfig.effects3D.enabled) {
            $('#effects3DControls, #effects3DParallaxContainer, #effects3DRotationContainer').show();
        } else {
            $('#effects3DControls, #effects3DParallaxContainer, #effects3DRotationContainer').hide();
        }
        
        // Real-Time Sync controls
        if (!currentConfig.realTimeSync) {
            currentConfig.realTimeSync = {
                enabled: true,
                roomId: 'demo-room-xtremecode',
                serverUrl: null,
                showViewerCount: true,
                showConnectionStatus: false,
                syncInterval: 1000,
                reconnectInterval: 3000,
                maxReconnectAttempts: 5,
                useLocalStorageFallback: true
            };
        }
        $('#realTimeSyncEnabled').prop('checked', currentConfig.realTimeSync.enabled !== false);
        $('#showViewerCount').prop('checked', currentConfig.realTimeSync.showViewerCount !== false);
        if (currentConfig.realTimeSync.enabled !== false) {
            $('#realTimeSyncControls').show();
        } else {
            $('#realTimeSyncControls').hide();
        }
        
        // Behavioral Targeting controls
        if (!currentConfig.behavioralTargeting) {
            currentConfig.behavioralTargeting = {
                enabled: false,
                rules: [
                    {
                        condition: 'newVisitor',
                        duration: 86400,
                        message: 'Offerta valida per 24 ore!'
                    },
                    {
                        condition: 'returningVisitor',
                        duration: 7200,
                        message: 'Ultime 2 ore! Non perdere l\'occasione!'
                    },
                    {
                        condition: 'mobile',
                        duration: 3600,
                        message: 'Solo 1 ora rimasta!'
                    },
                    {
                        condition: 'desktop',
                        duration: 86400,
                        message: 'Offerta valida per 24 ore'
                    },
                    {
                        condition: 'socialMedia',
                        duration: 3600,
                        message: 'Offerta limitata! Solo 1 ora!'
                    },
                    {
                        condition: 'email',
                        duration: 21600,
                        message: 'Offerta valida per 6 ore'
                    },
                    {
                        condition: 'direct',
                        duration: 86400,
                        message: 'Offerta valida per 24 ore'
                    }
                ],
                priority: ['newVisitor', 'returningVisitor', 'device', 'source'],
                storageKey: 'countdown_timer_visits',
                visitThreshold: 2
            };
        }
        $('#behavioralTargetingEnabled').prop('checked', currentConfig.behavioralTargeting.enabled === true);
        if (currentConfig.behavioralTargeting.enabled === true) {
            $('#behavioralTargetingControls').show();
        } else {
            $('#behavioralTargetingControls').hide();
        }
        
        // AI Optimization controls
        if (!currentConfig.aiOptimization) {
            currentConfig.aiOptimization = {
                enabled: false,
                trackInteractions: true,
                trackConversions: true,
                learningRate: 0.1,
                minSamples: 10,
                optimizationInterval: 3600000,
                storageKey: 'countdown_timer_ai_data'
            };
        }
        $('#aiOptimizationEnabled').prop('checked', currentConfig.aiOptimization.enabled === true);
        $('#trackInteractions').prop('checked', currentConfig.aiOptimization.trackInteractions !== false);
        $('#trackConversions').prop('checked', currentConfig.aiOptimization.trackConversions !== false);
        if (currentConfig.aiOptimization.enabled === true) {
            $('#aiOptimizationControls').show();
        } else {
            $('#aiOptimizationControls').hide();
        }
        
    }
    
    
    /**
     * Update preview
     */
    /**
     * Update the live preview with current configuration
     * Destroys existing timer instance and creates a new one with updated options
     */
    // Debounce updatePreview to prevent excessive destroy/recreate cycles
    let updatePreviewTimeout = null;
    let isUpdatingPreview = false;
    
    function updatePreview() {
        const $preview = $('#timer-preview');
        
        if (!$preview.length || !$preview.is(':visible')) {
            return;
        }
        
        // Debounce: cancel previous update if still pending
        if (updatePreviewTimeout) {
            clearTimeout(updatePreviewTimeout);
            updatePreviewTimeout = null;
        }
        
        // If already updating, queue this update with longer delay to prevent accumulation
        if (isUpdatingPreview) {
            updatePreviewTimeout = setTimeout(updatePreview, 200);
            return;
        }
        
        isUpdatingPreview = true;
        
        // CRITICAL: Force garbage collection by clearing any pending operations
        // Stop any running animations from previous instance
        const existingInstance = $preview.data('countdownTimerPro');
        if (existingInstance) {
            // Stop immediately without waiting
            if (typeof existingInstance.stop === 'function') {
                existingInstance.stop();
            }
            if (typeof existingInstance.stopCelebration === 'function') {
                existingInstance.stopCelebration();
            }
        }
        
        // Preserve liquidWaveOffset from existing instance to maintain wave animation continuity
        // This is ESPECIALLY important for luxury-gold and dark-neon presets where multicolor is active
        // The wave animation must be COMPLETELY ISOLATED from other animations (multicolor, pulseGlow, gradientWave)
        let preservedLiquidWaveOffset = null;
        
        // Destroy existing instance if it exists
        try {
            const existingInstance = $preview.data('countdownTimerPro');
            if (existingInstance) {
                // Preserve liquidWaveOffset to maintain continuous wave animation when changing presets
                // This prevents waves from disappearing/appearing when preset changes (especially luxury-gold and dark-neon)
                // liquidWaveOffset is INDEPENDENT and must NOT be affected by multicolor or other animations
                if (existingInstance.liquidWaveOffset !== undefined && 
                    existingInstance.liquidWaveOffset !== null && 
                    !isNaN(existingInstance.liquidWaveOffset) &&
                    isFinite(existingInstance.liquidWaveOffset)) {
                    // Always preserve the value, even if it's 0 (it might be at the start of animation)
                    preservedLiquidWaveOffset = existingInstance.liquidWaveOffset;
                }
                
                // CRITICAL: Clean up localStorage BEFORE destroying to prevent counting old instances
                if (typeof existingInstance.cleanupLocalStorageSync === 'function') {
                    existingInstance.cleanupLocalStorageSync();
                }
                
                if (typeof existingInstance.destroy === 'function') {
                    existingInstance.destroy();
                }
                // Force reset startTime to ensure timer starts from 0
                if (existingInstance.startTime !== undefined) {
                    existingInstance.startTime = null;
                }
                $preview.removeData('countdownTimerPro');
            }
        } catch (e) {
            // Ignore errors during destroy
        }
        
        // Clear the element and reset timer instance
        $preview.empty();
        timerInstance = null;
        
        // Small delay to ensure DOM is ready and previous instance is fully destroyed
        setTimeout(function() {
            if (!$preview.length || !$preview.is(':visible')) {
                return;
            }
            
            // Prepare multicolor config with validation
            const multicolorConfig = $.extend({}, currentConfig.multicolor);
            if (multicolorConfig.enabled && multicolorConfig.colors) {
                // Validate and filter colors
                multicolorConfig.colors = multicolorConfig.colors.filter(c => {
                    return /^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/.test(c);
                });
                // If no valid colors, disable multicolor
                if (multicolorConfig.colors.length === 0) {
                    multicolorConfig.enabled = false;
                    multicolorConfig.colors = ['#667eea'];
                }
            }
            
            // Get actual container dimensions for responsive preview
            const $container = $preview.parent();
            let actualWidth = currentConfig.width;
            let actualHeight = currentConfig.height;
            
            // Check if responsive mode is active
            const $activeBtn = $('.responsive-btn.active');
            if ($activeBtn.length) {
                const widthData = $activeBtn.data('width');
                if (widthData && widthData !== 'auto') {
                    const widthNum = parseInt(widthData);
                    actualWidth = widthNum;
                    // Calculate proportional height based on width
                    if (widthNum <= 400) {
                        // Mobile - ensure sufficient height for vertical layout
                        actualHeight = Math.max(700, widthNum * 2.8);
                    } else if (widthNum <= 768) {
                        // Tablet - maintain aspect ratio
                        actualHeight = Math.max(450, Math.round(widthNum * 0.5));
                    } else if (widthNum <= 1024) {
                        // Desktop - maintain aspect ratio
                        actualHeight = Math.max(500, Math.round(widthNum * 0.55));
                    } else {
                        // Large screens - use more height for better visibility
                        actualHeight = Math.max(600, Math.min(800, Math.round(widthNum * 0.45)));
                    }
                } else {
                    // Auto mode - use container dimensions
                    actualWidth = $container.width() || currentConfig.width || 900;
                    actualHeight = $container.height() || currentConfig.height || 450;
                }
            } else {
                // Use container dimensions if available
                actualWidth = $container.width() || currentConfig.width || 900;
                actualHeight = $container.height() || currentConfig.height || 450;
            }
            
            // Prepare celebration config
            const celebrationConfig = currentConfig.celebration ? $.extend({}, currentConfig.celebration) : {
                enabled: true,
                type: 'both',
                intensity: 1.0,
                colors: ['#667eea', '#764ba2', '#f093fb', '#4facfe', '#43e97b', '#fa709a', '#fee140', '#30cfd0']
            };
            
            // Prepare pulse/glow config
            const pulseGlowConfig = currentConfig.pulseGlow ? $.extend({}, currentConfig.pulseGlow) : {
                enabled: false,
                intensity: 1.0,
                speed: 1.0
            };
            
            // Prepare gradient wave config
            const gradientWaveConfig = currentConfig.gradientWave ? $.extend({}, currentConfig.gradientWave) : {
                enabled: false,
                colors: ['#667eea', '#764ba2', '#f093fb', '#4facfe'],
                speed: 1.0
            };
            
            // Prepare sound config
            const soundConfig = currentConfig.sound ? $.extend({}, currentConfig.sound) : {
                enabled: false,
                tickEnabled: true,
                finishEnabled: true,
                tickVolume: 0.3,
                finishVolume: 0.5,
                tickFrequency: 800,
                finishFrequency: 600
            };
            
            // Prepare 3D effects config
            const effects3DConfig = currentConfig.effects3D ? $.extend({}, currentConfig.effects3D) : {
                enabled: false,
                shadowDepth: 20,
                rotationSpeed: 0,
                parallaxIntensity: 0,
                lightingIntensity: 0.5
            };
            
            // Prepare liquid effect config (Distinctive visual feature - ALWAYS enabled, high definition)
            // CRITICAL: Liquid effect is the default visual style - always include it
            const liquidEffectConfig = currentConfig.liquidEffect ? $.extend({}, currentConfig.liquidEffect) : {
                enabled: true, // ALWAYS enabled - distinctive visual feature
                intensity: 1.2, // High definition
                bubbleSpeed: 1.0,
                waveSpeed: 1.0,
                bubbleSpawnRate: 0.8, // Higher rate for more realistic effect
                showEvaporation: true,
                liquidColor: null
            };
            // FORCE enabled to true - liquid effect is ALWAYS active, never disabled
            liquidEffectConfig.enabled = true;
            
            // Prepare cross-tab sync config (FEATURE 1)
            const realTimeSyncConfig = currentConfig.realTimeSync ? $.extend({}, currentConfig.realTimeSync) : {
                enabled: true, // Enable by default for demo
                roomId: 'demo-room-xtremecode', // Same room ID for all demo instances
                serverUrl: null, // null = use localStorage fallback
                showViewerCount: true, // Show "X people watching" (default: enabled)
                showConnectionStatus: false, // Show connection status (disabled by default)
                syncInterval: 1000, // Sync every 1 second
                reconnectInterval: 3000,
                maxReconnectAttempts: 5,
                useLocalStorageFallback: true // Use localStorage if WebSocket unavailable
            };
            
            // Behavioral Targeting config (FEATURE 2)
            const behavioralTargetingConfig = currentConfig.behavioralTargeting ? $.extend({}, currentConfig.behavioralTargeting) : {
                enabled: false,
                rules: [],
                priority: ['newVisitor', 'returningVisitor', 'device', 'source'],
                storageKey: 'countdown_timer_visits',
                visitThreshold: 2
            };
            
            // Conversion Optimization config (FEATURE 3)
            const aiOptimizationConfig = currentConfig.aiOptimization ? $.extend({}, currentConfig.aiOptimization) : {
                enabled: false,
                trackInteractions: true,
                trackConversions: true,
                learningRate: 0.1,
                minSamples: 10,
                optimizationInterval: 3600000,
                storageKey: 'countdown_timer_ai_data'
            };
            
            // Create options object
            const options = {
                timeType: currentConfig.timeType,
                duration: currentConfig.duration,
                showDays: currentConfig.showDays,
                showHours: currentConfig.showHours,
                showMinutes: currentConfig.showMinutes,
                showSeconds: currentConfig.showSeconds,
                width: actualWidth,
                height: actualHeight,
                fontSize: currentConfig.fontSize,
                circleThickness: currentConfig.circleThickness,
                colors: $.extend({}, currentConfig.colors),
                fontFamily: currentConfig.fontFamily,
                fontWeight: currentConfig.fontWeight,
                labelFontWeight: currentConfig.labelFontWeight,
                multicolor: multicolorConfig,
                celebration: celebrationConfig,
                pulseGlow: pulseGlowConfig,
                gradientWave: gradientWaveConfig,
                sound: soundConfig,
                effects3D: effects3DConfig,
                liquidEffect: liquidEffectConfig, // Distinctive visual feature - Liquid that evaporates with bubbles
                realTimeSync: realTimeSyncConfig, // FEATURE 1: Cross-Tab Synchronization with optional WebSocket server support
                behavioralTargeting: behavioralTargetingConfig, // FEATURE 2: Behavioral Targeting Engine
                aiOptimization: aiOptimizationConfig // FEATURE 3: Conversion Optimization (Local A/B Testing)
            };
            
            // For countup mode, duration represents the maximum value where timer stops
            // Timer starts from 0 and stops when it reaches the duration
            // startTime is set to current time (or null to use current time)
            if (currentConfig.timeType === 'countup') {
                // Don't set startTime, let it default to current time (starts from 0)
                // Duration will be used as the maximum limit
                // Make sure duration is a positive number
                if (!currentConfig.duration || currentConfig.duration <= 0) {
                    currentConfig.duration = 3600; // Default to 1 hour if invalid
                }
            }
            
            // CRITICAL: Set preserved liquidWaveOffset in options BEFORE creating timer
            // This ensures the wave offset is available from the start, preventing waves from disappearing
            if (preservedLiquidWaveOffset !== null && 
                !isNaN(preservedLiquidWaveOffset) && 
                isFinite(preservedLiquidWaveOffset) &&
                preservedLiquidWaveOffset >= 0) {
                // Store in options so it can be accessed in constructor
                options._preservedLiquidWaveOffset = preservedLiquidWaveOffset;
            }
            
            // Initialize timer
            try {
                $preview.countdownTimerPro(options);
                timerInstance = $preview.data('countdownTimerPro');
                
                // CRITICAL: RESTORE preserved liquidWaveOffset IMMEDIATELY after creation
                // This ensures waves never disappear when changing presets
                if (timerInstance) {
                    // FORCE liquidEffect to always be enabled - it's the default visual style
                    if (!timerInstance.options.liquidEffect) {
                        timerInstance.options.liquidEffect = {
                            enabled: true,
                            intensity: 1.2,
                            bubbleSpeed: 1.0,
                            waveSpeed: 1.0,
                            bubbleSpawnRate: 0.8,
                            showEvaporation: true,
                            liquidColor: null
                        };
                    }
                    timerInstance.options.liquidEffect.enabled = true; // ALWAYS enabled
                    
                    // CRITICAL: Restore preserved wave offset IMMEDIATELY and SYNCHRONOUSLY
                    // This must happen before the first draw to prevent waves from disappearing
                    // ESPECIALLY important for luxury-gold and dark-neon where multicolor is active
                    if (preservedLiquidWaveOffset !== null && 
                        !isNaN(preservedLiquidWaveOffset) && 
                        isFinite(preservedLiquidWaveOffset)) {
                        // Always restore, even if value is 0 (it might be at animation start)
                        timerInstance.liquidWaveOffset = preservedLiquidWaveOffset;
                    } else if (options._preservedLiquidWaveOffset !== undefined && 
                               !isNaN(options._preservedLiquidWaveOffset) && 
                               isFinite(options._preservedLiquidWaveOffset)) {
                        // Use value from options if available
                        timerInstance.liquidWaveOffset = options._preservedLiquidWaveOffset;
                    } else {
                        // Initialize if not preserved or invalid
                        if (timerInstance.liquidWaveOffset === undefined || 
                            timerInstance.liquidWaveOffset === null || 
                            isNaN(timerInstance.liquidWaveOffset) ||
                            !isFinite(timerInstance.liquidWaveOffset)) {
                            timerInstance.liquidWaveOffset = 0;
                        }
                    }
                    
                    // CRITICAL: Force immediate update to ensure wave offset is applied before first draw
                    // Use multiple requestAnimationFrame calls to ensure it happens before the first render
                    // This is especially important when multicolor is active (luxury-gold, dark-neon)
                    requestAnimationFrame(function() {
                        if (timerInstance && preservedLiquidWaveOffset !== null && 
                            !isNaN(preservedLiquidWaveOffset) && 
                            isFinite(preservedLiquidWaveOffset)) {
                            timerInstance.liquidWaveOffset = preservedLiquidWaveOffset;
                        }
                        // Double-check after another frame to ensure it's set
                        requestAnimationFrame(function() {
                            if (timerInstance && preservedLiquidWaveOffset !== null && 
                                !isNaN(preservedLiquidWaveOffset) && 
                                isFinite(preservedLiquidWaveOffset)) {
                                timerInstance.liquidWaveOffset = preservedLiquidWaveOffset;
                            }
                        });
                    });
                    
                    // Reset updating flag after successful timer creation
                    isUpdatingPreview = false;
                }
                
            } catch (e) {
                // Reset updating flag on error
                isUpdatingPreview = false;
                // Silently retry initialization after a short delay
                setTimeout(function() {
                    if ($preview.length && $preview.is(':visible')) {
                        try {
                            // CRITICAL: Ensure preserved liquidWaveOffset is in options before retry
                            if (preservedLiquidWaveOffset !== null && 
                                !isNaN(preservedLiquidWaveOffset) && 
                                isFinite(preservedLiquidWaveOffset) &&
                                preservedLiquidWaveOffset >= 0) {
                                options._preservedLiquidWaveOffset = preservedLiquidWaveOffset;
                            }
                            
                            $preview.countdownTimerPro(options);
                            timerInstance = $preview.data('countdownTimerPro');
                            
                            // CRITICAL: RESTORE preserved liquidWaveOffset and FORCE liquidEffect enabled
                            if (timerInstance) {
                                // Force liquidEffect to always be enabled
                                if (!timerInstance.options.liquidEffect) {
                                    timerInstance.options.liquidEffect = {
                                        enabled: true,
                                        intensity: 1.2,
                                        bubbleSpeed: 1.0,
                                        waveSpeed: 1.0,
                                        bubbleSpawnRate: 0.8,
                                        showEvaporation: true,
                                        liquidColor: null
                                    };
                                }
                                timerInstance.options.liquidEffect.enabled = true;
                                
                                // CRITICAL: Restore preserved wave offset IMMEDIATELY
                                if (preservedLiquidWaveOffset !== null && 
                                    !isNaN(preservedLiquidWaveOffset) && 
                                    isFinite(preservedLiquidWaveOffset) &&
                                    preservedLiquidWaveOffset >= 0) {
                                    timerInstance.liquidWaveOffset = preservedLiquidWaveOffset;
                                } else if (options._preservedLiquidWaveOffset !== undefined) {
                                    timerInstance.liquidWaveOffset = options._preservedLiquidWaveOffset;
                                } else if (timerInstance.liquidWaveOffset === undefined || 
                                          timerInstance.liquidWaveOffset === null || 
                                          isNaN(timerInstance.liquidWaveOffset) ||
                                          !isFinite(timerInstance.liquidWaveOffset)) {
                                    timerInstance.liquidWaveOffset = 0;
                                }
                                
                                // CRITICAL: Force immediate update to ensure wave offset is applied
                                requestAnimationFrame(function() {
                                    if (timerInstance && preservedLiquidWaveOffset !== null && 
                                        !isNaN(preservedLiquidWaveOffset) && 
                                        isFinite(preservedLiquidWaveOffset)) {
                                        timerInstance.liquidWaveOffset = preservedLiquidWaveOffset;
                                    }
                                });
                            }
                            
                        } catch (e2) {
                            // Silently fail - timer may not be available
                        }
                    }
                }, 200);
            }
            
            // CRITICAL: Reset updating flag after setTimeout completes
            // This ensures the flag is reset even if timer creation fails silently
            setTimeout(function() {
                isUpdatingPreview = false;
            }, 200);
        }, 50);
    }
    
    /**
     * Generate code output
     */
    /**
     * Update the generated code output display
     * Formats and displays the JavaScript code for the current configuration
     */
    function updateCodeOutput() {
        const code = generateCode();
        // Use text() instead of html() for CSP compliance - formatCode already escapes HTML
        const formattedCode = formatCode(code);
        $('#codeOutput').html(formattedCode);
    }
    
    /**
     * Generate JavaScript code
     */
    /**
     * Generate JavaScript code for the current configuration
     * @returns {string} Complete JavaScript code to initialize the timer
     */
    function generateCode() {
        let code = "$('#my-timer').countdownTimerPro({\n";
        
        // Basic options
        code += `    timeType: '${currentConfig.timeType}',\n`;
        code += `    duration: ${currentConfig.duration}, // For countup: maximum value where timer stops\n`;
        
        // Display options
        code += `    showDays: ${currentConfig.showDays},\n`;
        code += `    showHours: ${currentConfig.showHours},\n`;
        code += `    showMinutes: ${currentConfig.showMinutes},\n`;
        code += `    showSeconds: ${currentConfig.showSeconds},\n`;
        
        // Size options
        code += `    width: ${currentConfig.width},\n`;
        code += `    height: ${currentConfig.height},\n`;
        code += `    fontSize: ${currentConfig.fontSize},\n`;
        code += `    circleThickness: ${currentConfig.circleThickness},\n`;
        
        // Color options
        code += `    colors: {\n`;
        code += `        circle: '${currentConfig.colors.circle}',\n`;
        code += `        circleBackground: '${currentConfig.colors.circleBackground}',\n`;
        code += `        text: '${currentConfig.colors.text}'\n`;
        code += `    },\n`;
        
        // Font options
        code += `    fontFamily: '${currentConfig.fontFamily}',\n`;
        code += `    fontWeight: '${currentConfig.fontWeight}',\n`;
        code += `    labelFontWeight: '${currentConfig.labelFontWeight}',\n`;
        
        // Multicolor
        if (currentConfig.multicolor && currentConfig.multicolor.enabled) {
            const multicolorColors = (currentConfig.multicolor.colors || []).filter(c => /^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/.test(c));
            if (multicolorColors.length === 0) {
                multicolorColors.push('#667eea');
            }
            code += `    multicolor: {\n`;
            code += `        enabled: true,\n`;
            code += `        colors: [${multicolorColors.map(c => `'${c}'`).join(', ')}]\n`;
            code += `    },\n`;
        } else {
            code += `    multicolor: {\n`;
            code += `        enabled: false\n`;
            code += `    },\n`;
        }
        
        // Celebration
        if (currentConfig.celebration && currentConfig.celebration.enabled) {
            code += `    celebration: {\n`;
            code += `        enabled: true,\n`;
            code += `        type: '${currentConfig.celebration.type || 'both'}',\n`;
            code += `        intensity: ${currentConfig.celebration.intensity || 1.0}\n`;
            code += `    },\n`;
        } else {
            code += `    celebration: {\n`;
            code += `        enabled: false\n`;
            code += `    },\n`;
        }
        
        // Pulse/Glow
        if (currentConfig.pulseGlow && currentConfig.pulseGlow.enabled) {
            code += `    pulseGlow: {\n`;
            code += `        enabled: true,\n`;
            code += `        intensity: ${currentConfig.pulseGlow.intensity || 1.0},\n`;
            code += `        speed: ${currentConfig.pulseGlow.speed || 1.0}\n`;
            code += `    },\n`;
        } else {
            code += `    pulseGlow: {\n`;
            code += `        enabled: false\n`;
            code += `    },\n`;
        }
        
        // Gradient Wave
        if (currentConfig.gradientWave && currentConfig.gradientWave.enabled) {
            const gradientColors = (currentConfig.gradientWave.colors || []).filter(c => /^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/.test(c));
            if (gradientColors.length === 0) {
                gradientColors.push('#667eea', '#764ba2', '#f093fb', '#4facfe');
            }
            code += `    gradientWave: {\n`;
            code += `        enabled: true,\n`;
            code += `        colors: [${gradientColors.map(c => `'${c}'`).join(', ')}],\n`;
            code += `        speed: ${currentConfig.gradientWave.speed || 1.0}\n`;
            code += `    },\n`;
        }
        
        // Liquid Effect (Distinctive visual feature - always enabled, high definition)
        // Always include liquid effect - it's the default visual style
        code += `    liquidEffect: {\n`;
        code += `        enabled: true, // Always enabled - distinctive high definition liquid effect\n`;
        code += `        intensity: ${(currentConfig.liquidEffect && currentConfig.liquidEffect.intensity) || 1.2},\n`;
        code += `        bubbleSpeed: ${(currentConfig.liquidEffect && currentConfig.liquidEffect.bubbleSpeed) || 1.0},\n`;
        code += `        waveSpeed: ${(currentConfig.liquidEffect && currentConfig.liquidEffect.waveSpeed) || 1.0},\n`;
        code += `        bubbleSpawnRate: ${(currentConfig.liquidEffect && currentConfig.liquidEffect.bubbleSpawnRate) || 0.8},\n`;
        code += `        showEvaporation: ${(currentConfig.liquidEffect && currentConfig.liquidEffect.showEvaporation !== false) || true},\n`;
        if (currentConfig.liquidEffect && currentConfig.liquidEffect.liquidColor) {
            code += `        liquidColor: '${currentConfig.liquidEffect.liquidColor}',\n`;
        }
        code += `    },\n`;
        
        // Sound Effects
        if (currentConfig.sound && currentConfig.sound.enabled) {
            code += `    sound: {\n`;
            code += `        enabled: true,\n`;
            code += `        tickEnabled: ${currentConfig.sound.tickEnabled !== undefined ? currentConfig.sound.tickEnabled : true},\n`;
            code += `        finishEnabled: ${currentConfig.sound.finishEnabled !== undefined ? currentConfig.sound.finishEnabled : true},\n`;
            code += `        tickVolume: ${currentConfig.sound.tickVolume || 0.3},\n`;
            code += `        finishVolume: ${currentConfig.sound.finishVolume || 0.5},\n`;
            code += `        tickFrequency: ${currentConfig.sound.tickFrequency || 800},\n`;
            code += `        finishFrequency: ${currentConfig.sound.finishFrequency || 600}\n`;
            code += `    },\n`;
        } else {
            code += `    sound: {\n`;
            code += `        enabled: false\n`;
            code += `    },\n`;
        }
        
        // 3D Effects
        if (currentConfig.effects3D && currentConfig.effects3D.enabled) {
            code += `    effects3D: {\n`;
            code += `        enabled: true,\n`;
            code += `        shadowDepth: ${currentConfig.effects3D.shadowDepth || 20},\n`;
            code += `        rotationSpeed: ${currentConfig.effects3D.rotationSpeed || 0},\n`;
            code += `        parallaxIntensity: ${currentConfig.effects3D.parallaxIntensity || 0},\n`;
            code += `        lightingIntensity: ${currentConfig.effects3D.lightingIntensity || 0.5}\n`;
            code += `    },\n`;
        } else {
            code += `    effects3D: {\n`;
            code += `        enabled: false\n`;
            code += `    },\n`;
        }
        
        // Cross-Tab Synchronization (FEATURE 1: Cross-tab sync with optional WebSocket server support)
        if (currentConfig.realTimeSync && currentConfig.realTimeSync.enabled !== false) {
            const roomId = currentConfig.realTimeSync.roomId ? `'${currentConfig.realTimeSync.roomId}'` : 'null';
            const serverUrl = currentConfig.realTimeSync.serverUrl ? `'${currentConfig.realTimeSync.serverUrl}'` : 'null';
            code += `    realTimeSync: {\n`;
            code += `        enabled: true,\n`;
            code += `        roomId: ${roomId}, // Auto-generated if null\n`;
            code += `        serverUrl: ${serverUrl}, // null = use localStorage fallback\n`;
            code += `        showViewerCount: ${currentConfig.realTimeSync.showViewerCount !== false},\n`;
            code += `        showConnectionStatus: ${currentConfig.realTimeSync.showConnectionStatus === true},\n`;
            code += `        syncInterval: ${currentConfig.realTimeSync.syncInterval || 1000},\n`;
            code += `        reconnectInterval: ${currentConfig.realTimeSync.reconnectInterval || 3000},\n`;
            code += `        maxReconnectAttempts: ${currentConfig.realTimeSync.maxReconnectAttempts || 5},\n`;
            code += `        useLocalStorageFallback: ${currentConfig.realTimeSync.useLocalStorageFallback !== false}\n`;
            code += `    },\n`;
        } else {
            code += `    realTimeSync: {\n`;
            code += `        enabled: false\n`;
            code += `    },\n`;
        }
        
        // Behavioral Targeting (FEATURE 2: Timer that adapts based on user behavior)
        if (currentConfig.behavioralTargeting && currentConfig.behavioralTargeting.enabled === true) {
            const rules = currentConfig.behavioralTargeting.rules || [];
            code += `    behavioralTargeting: {\n`;
            code += `        enabled: true,\n`;
            code += `        rules: [\n`;
            rules.forEach((rule, index) => {
                const duration = rule.duration || 86400;
                const message = rule.message ? `'${rule.message.replace(/'/g, "\\'")}'` : 'null';
                code += `            { condition: '${rule.condition}', duration: ${duration}, message: ${message} }${index < rules.length - 1 ? ',' : ''}\n`;
            });
            code += `        ],\n`;
            code += `        priority: [${currentConfig.behavioralTargeting.priority.map(p => `'${p}'`).join(', ')}],\n`;
            code += `        storageKey: '${currentConfig.behavioralTargeting.storageKey || 'countdown_timer_visits'}',\n`;
            code += `        visitThreshold: ${currentConfig.behavioralTargeting.visitThreshold || 2}\n`;
            code += `    },\n`;
        } else {
            code += `    behavioralTargeting: {\n`;
            code += `        enabled: false\n`;
            code += `    },\n`;
        }
        
        // Conversion Optimization (FEATURE 3: Local A/B testing and automatic variant optimization)
        if (currentConfig.aiOptimization && currentConfig.aiOptimization.enabled === true) {
            code += `    aiOptimization: {\n`;
            code += `        enabled: true,\n`;
            code += `        trackInteractions: ${currentConfig.aiOptimization.trackInteractions !== false},\n`;
            code += `        trackConversions: ${currentConfig.aiOptimization.trackConversions !== false},\n`;
            code += `        learningRate: ${currentConfig.aiOptimization.learningRate || 0.1},\n`;
            code += `        minSamples: ${currentConfig.aiOptimization.minSamples || 10},\n`;
            code += `        optimizationInterval: ${currentConfig.aiOptimization.optimizationInterval || 3600000},\n`;
            code += `        storageKey: '${currentConfig.aiOptimization.storageKey || 'countdown_timer_ai_data'}'\n`;
            code += `    },\n`;
        } else {
            code += `    aiOptimization: {\n`;
            code += `        enabled: false\n`;
            code += `    },\n`;
        }
        
        code += "});";
        
        return code;
    }
    
    /**
     * Format code with syntax highlighting
     */
    /**
     * Format code with syntax highlighting
     * @param {string} code - Raw JavaScript code
     * @returns {string} HTML-formatted code with syntax highlighting
     */
    function formatCode(code) {
        // Escape HTML first
        let escaped = code
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;');
        
        // Apply syntax highlighting
        return escaped
            .replace(/(['"])([^'"]+)\1/g, '<span style="color: #95e1d3;">$1$2$1</span>')
            .replace(/\b(\d+)\b/g, '<span style="color: #f38181;">$1</span>')
            .replace(/(\w+):/g, '<span style="color: #fce38a;">$1</span>:');
    }
    
    /**
     * Copy code to clipboard
     */
    /**
     * Copy generated code to clipboard
     * Uses document.execCommand for clipboard access
     */
    function copyCodeToClipboard() {
        const code = generateCode();
        const textarea = $('<textarea>').val(code).appendTo('body').select();
        
        try {
            document.execCommand('copy');
            const $btn = $('#copyBtn');
            const originalText = $btn.text();
            $btn.addClass('copied').text('Copied!').css('background', '#28a745');
            setTimeout(() => {
                $btn.removeClass('copied').text(originalText).css('background', '#667eea');
            }, 2000);
        } catch (err) {
            alert('Failed to copy code. Please select and copy manually.');
        }
        
        textarea.remove();
    }
    
    /**
     * Export preset as JSON file
     */
    /**
     * Export current configuration as JSON preset file
     * Prompts user for preset name and downloads JSON file
     */
    function exportPreset() {
        const presetData = {
            name: prompt('Enter a name for this preset:', 'My Custom Preset') || 'My Custom Preset',
            version: '1.0.0',
            timestamp: new Date().toISOString(),
            config: $.extend(true, {}, currentConfig)
        };
        
        const dataStr = JSON.stringify(presetData, null, 2);
        const dataBlob = new Blob([dataStr], { type: 'application/json' });
        const url = URL.createObjectURL(dataBlob);
        const link = document.createElement('a');
        link.href = url;
        link.download = presetData.name.replace(/[^a-z0-9]/gi, '_').toLowerCase() + '.json';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        URL.revokeObjectURL(url);
        
        $('#exportPresetBtn').text('✓ Exported').css('background', '#28a745');
        setTimeout(() => {
            $('#exportPresetBtn').text('💾 Export').css('background', '#667eea');
        }, 2000);
    }
    
    /**
     * Load preset from configuration object
     */
    /**
     * Load preset from configuration object
     * @param {Object} config - Configuration object or preset file with metadata
     */
    function loadPresetFromConfig(config) {
        if (config.config) {
            // If it's a preset file with metadata
            currentConfig = $.extend(true, {}, config.config);
        } else {
            // If it's just the config object
            currentConfig = $.extend(true, {}, config);
        }
        
        updateControlsFromConfig();
        updatePreview();
        updateCodeOutput();
    }
    
    // Make function globally available for URL loading
    window.loadPresetFromConfig = loadPresetFromConfig;
    
    /**
     * Share configuration via URL
     */
    /**
     * Share configuration via URL
     * Encodes current configuration in URL parameter and copies to clipboard
     */
    function shareViaUrl() {
        const configStr = JSON.stringify(currentConfig);
        const encoded = encodeURIComponent(configStr);
        const url = window.location.origin + window.location.pathname + '?config=' + encoded;
        
        // Copy to clipboard
        const textarea = $('<textarea>').val(url).appendTo('body').select();
        try {
            document.execCommand('copy');
            $('#shareUrlBtn').text('✓ URL Copied!').css('background', '#28a745');
            setTimeout(() => {
                $('#shareUrlBtn').text('🔗 Share via URL').css('background', '#28a745');
            }, 3000);
            
            // Also show in alert with option to open
            if (confirm('URL copied to clipboard!\n\nClick OK to open in new tab, or Cancel to keep working here.')) {
                window.open(url, '_blank');
            }
        } catch (err) {
            alert('URL:\n\n' + url + '\n\nPlease copy this URL manually.');
        }
        textarea.remove();
    }
    
    /**
     * Export timer as PNG image
     */
    /**
     * Export timer preview as PNG image
     * Creates high-resolution canvas and downloads as PNG file
     */
    function exportAsImage() {
        const $preview = $('#timer-preview');
        if (!$preview.length) {
            alert('Timer preview not found.');
            return;
        }
        
        const canvas = $preview.find('canvas')[0];
        if (!canvas) {
            alert('Timer canvas not found. Please wait for the timer to load.');
            return;
        }
        
        try {
            // Get timer instance to get the actual canvas
            const timerInstance = $preview.data('countdownTimerPro');
            if (!timerInstance || !timerInstance.canvas) {
                alert('Timer instance not found. Please wait for the timer to load.');
                return;
            }
            
            const sourceCanvas = timerInstance.canvas;
            
            // Get the actual internal canvas size (high resolution, not CSS size)
            // The canvas is already rendered at high DPI, so we use its internal dimensions
            const internalWidth = sourceCanvas.width;
            const internalHeight = sourceCanvas.height;
            
            // For even better quality, we'll render at 6x the display size
            // This ensures crisp output even for very large displays
            const displayWidth = parseInt($preview.css('width')) || internalWidth / (window.devicePixelRatio || 1);
            const displayHeight = parseInt($preview.css('height')) || internalHeight / (window.devicePixelRatio || 1);
            
            // Use the internal canvas size as base, then scale up for export
            // This preserves all the detail from the high-DPI rendering
            const exportScale = 6; // 6x for ultra-high quality (equivalent to 600 DPI)
            const exportCanvas = document.createElement('canvas');
            exportCanvas.width = displayWidth * exportScale;
            exportCanvas.height = displayHeight * exportScale;
            const ctx = exportCanvas.getContext('2d');
            
            // Enable highest quality settings
            ctx.imageSmoothingEnabled = true;
            ctx.imageSmoothingQuality = 'high';
            
            // Fill with dark background to match preview
            ctx.fillStyle = '#0a0a1a';
            ctx.fillRect(0, 0, exportCanvas.width, exportCanvas.height);
            
            // Draw the source canvas at high resolution
            // Use the internal high-res canvas directly
            ctx.save();
            ctx.scale(exportScale, exportScale);
            ctx.drawImage(sourceCanvas, 0, 0, displayWidth, displayHeight);
            ctx.restore();
            
            // Convert to blob with maximum quality
            exportCanvas.toBlob(function(blob) {
                if (!blob) {
                    alert('Error creating image. Please try again.');
                    return;
                }
                const url = URL.createObjectURL(blob);
                const link = document.createElement('a');
                link.href = url;
                link.download = 'countdown-timer-' + Date.now() + '.png';
                document.body.appendChild(link);
                link.click();
                document.body.removeChild(link);
                URL.revokeObjectURL(url);
                
                $('#exportImageBtn').text('✓ Exported').css('background', '#28a745');
                setTimeout(() => {
                    $('#exportImageBtn').text('📥 Export PNG').css('background', '#28a745');
                }, 2000);
            }, 'image/png', 1.0); // Maximum quality (1.0)
        } catch (err) {
            alert('Error exporting image. Please try again.');
        }
    }
    
})(jQuery);

