/**
 * Countdown Timer Pro - Advanced HTML5 Canvas Countdown Timer
 * Version: 1.0.0
 * Author: XtremeCode
 * 
 * A modern, feature-rich countdown/countup timer with beautiful animations
 * and extensive customization options.
 */

(function($) {
    'use strict';

    /**
     * CountdownTimerPro - Main timer class
     * Handles initialization, rendering, animations, and all timer functionality
     */
    class CountdownTimerPro {
        constructor(element, options) {
            this.$element = $(element);
            this.element = element;
            this.options = $.extend({}, CountdownTimerPro.defaults, options);
            
            // CRITICAL: Ensure liquidEffect is ALWAYS enabled - it's the default visual style
            if (!this.options.liquidEffect) {
                this.options.liquidEffect = {
                    enabled: true,
                    intensity: 1.2,
                    bubbleSpeed: 1.0,
                    waveSpeed: 1.0,
                    bubbleSpawnRate: 0.8,
                    showEvaporation: true,
                    liquidColor: null
                };
            }
            // FORCE enabled to true - liquid effect is ALWAYS active, never disabled
            this.options.liquidEffect.enabled = true;
            
            this.canvas = null;
            this.ctx = null;
            this.animationFrame = null;
            this.isRunning = false;
            this.startTime = null;
            this.endTime = null;
            this.currentTime = null;
            this.remainingTime = {
                days: 0,
                hours: 0,
                minutes: 0,
                seconds: 0,
                total: 0
            };
            this.animationProgress = 0;
            this.colorIndex = 0;
            this.pulseProgress = 0;
            this.gradientWaveProgress = 0;
            // liquidWaveOffset is initialized separately and preserved independently
            // It should NOT be reset here - it's initialized in constructor and preserved across preset changes
            this.instanceId = 'timer_' + Math.random().toString(36).substr(2, 9);
            
            // Cross-tab synchronization (FEATURE 1)
            this.realTimeSync = {
                enabled: false,
                roomId: null,
                ws: null,
                wsUrl: null,
                viewerCount: 0,
                connectionStatus: 'disconnected', // 'connected', 'connecting', 'disconnected'
                syncInterval: null,
                reconnectAttempts: 0,
                lastSyncTime: null,
                syncData: null
            };
            
            // Behavioral targeting (FEATURE 2: Timer adapts based on user behavior)
            this.behavioralTargeting = {
                enabled: false,
                rules: [],
                priority: ['newVisitor', 'returningVisitor', 'device', 'source'],
                storageKey: 'countdown_timer_visits',
                visitThreshold: 2,
                appliedRule: null,
                visitCount: 0,
                lastVisit: null
            };
            
            // Conversion Optimization (FEATURE 3: Local A/B testing for automatic optimization)
            this.aiOptimization = {
                enabled: false,
                trackInteractions: true,
                trackConversions: true,
                learningRate: 0.1,
                minSamples: 10,
                optimizationInterval: 3600000, // 1 hour
                storageKey: 'countdown_timer_ai_data',
                interactions: [],
                conversions: [],
                variants: {},
                currentVariant: null,
                lastOptimization: null,
                stats: {
                    totalInteractions: 0,
                    totalConversions: 0,
                    conversionRate: 0,
                    bestVariant: null
                }
            };
            
            // Sound effects
            this.audioContext = null;
            this.soundEnabled = false;
            this.lastSecondPlayed = -1;
            
            // 3D effects
            this.rotation3D = 0;
            
            // Background pattern
            this.patternCanvas = null;
            this.patternCtx = null;
            
            // Celebration animation system
            this.celebrationCanvas = null;
            this.celebrationCtx = null;
            this.celebrationParticles = [];
            this.celebrationAnimationFrame = null;
            this.isCelebrating = false;
            
            // Liquid effect with bubbles system (Distinctive visual feature)
            this.liquidBubbles = {}; // Store bubbles per circle unit
            // Initialize liquidWaveOffset from options if preserved (prevents waves disappearing on preset change)
            this.liquidWaveOffset = (options._preservedLiquidWaveOffset !== undefined && 
                                     !isNaN(options._preservedLiquidWaveOffset) && 
                                     isFinite(options._preservedLiquidWaveOffset)) 
                                    ? options._preservedLiquidWaveOffset : 0;
            this.lastBubbleSpawn = {}; // Track last bubble spawn time per circle
            // Smooth liquid level interpolation for fluid animation
            this.liquidLevels = {}; // Store animated liquid levels per unit
            this.liquidLevelTargets = {}; // Store target liquid levels per unit
            this.evaporationParticles = {}; // Particles that escape the circle
            
            this.init();
        }

        /**
         * Initialize the timer instance
         * 
         * @description
         * Sets up all timer components including canvas, audio context, background
         * patterns, real-time synchronization, behavioral targeting, and AI optimization.
         * Starts the animation loop and handles window resize events.
         */
        init() {
            // Initialize audio context for sound effects
            // Always initialize audio context if sound is enabled, even if it requires user interaction
            if (this.options.sound && this.options.sound.enabled) {
                this.initAudio();
                // If audio context is suspended (requires user interaction), resume it on first user interaction
                if (this.audioContext && this.audioContext.state === 'suspended') {
                    const resumeAudio = () => {
                        this.audioContext.resume().then(() => {
                            document.removeEventListener('click', resumeAudio);
                            document.removeEventListener('touchstart', resumeAudio);
                        }).catch(() => {
                            // Ignore errors
                        });
                    };
                    document.addEventListener('click', resumeAudio, { once: true });
                    document.addEventListener('touchstart', resumeAudio, { once: true });
                }
            }
            
            // Create canvas element
            this.createCanvas();
            
            // Create background pattern if enabled
            if (this.options.backgroundPattern && this.options.backgroundPattern.enabled) {
                this.createBackgroundPattern();
            }
            
            // Calculate initial time
            this.calculateTime();
            
            // Initialize cross-tab synchronization if enabled (FEATURE 1)
            if (this.options.realTimeSync && this.options.realTimeSync.enabled) {
                this.initRealTimeSync();
            }
            
            // Initialize behavioral targeting if enabled (FEATURE 2)
            if (this.options.behavioralTargeting && this.options.behavioralTargeting.enabled) {
                this.initBehavioralTargeting();
            }
            
            // Initialize conversion optimization if enabled (FEATURE 3)
            if (this.options.aiOptimization && this.options.aiOptimization.enabled) {
                this.initAIOptimization();
            }
            
            // Setup auto-reset if enabled
            if (this.options.autoReset.enabled) {
                this.setupAutoReset();
            }
            
            // Initial resize after a short delay to ensure container is ready
            // Use requestAnimationFrame to prevent initial jitter
            requestAnimationFrame(() => {
                setTimeout(() => {
                    this.resize();
                }, 100);
            });
            
            // Start the timer
            this.start();
            
            // Handle window resize with debounce and throttling
            let resizeTimeout;
            let isResizing = false;
            
            const handleResize = () => {
                if (isResizing) return;
                isResizing = true;
                
                clearTimeout(resizeTimeout);
                resizeTimeout = setTimeout(() => {
                    this.resize();
                    // Update viewer count UI position on resize (for mobile/desktop switch)
                    if (this.realTimeSync && this.realTimeSync.enabled) {
                        this.updateViewerCountUI();
                    }
                    isResizing = false;
                }, 250); // Increased debounce to 250ms
            };
            
            $(window).on('resize.' + this.instanceId, handleResize);
            
            // Also listen for container resize (if ResizeObserver is available)
            // Use throttling to prevent excessive calls
            if (typeof ResizeObserver !== 'undefined') {
                let lastWidth = 0;
                let lastHeight = 0;
                
                this.resizeObserver = new ResizeObserver((entries) => {
                    for (let entry of entries) {
                        const { width, height } = entry.contentRect;
                        
                        // Only trigger if size changed significantly (more than 2px)
                        if (Math.abs(width - lastWidth) > 2 || Math.abs(height - lastHeight) > 2) {
                            lastWidth = width;
                            lastHeight = height;
                            handleResize();
                        }
                    }
                });
                this.resizeObserver.observe(this.element);
            }
            
            // Force initial resize after container is fully rendered
            // Also trigger resize on container size changes
            const checkResize = () => {
                const currentWidth = this.$element.width();
                const currentHeight = this.$element.height();
                if (currentWidth > 0 && currentHeight > 0) {
                    this.resize();
                }
            };
            
            // Check resize multiple times to catch all layout changes
            // On mobile, check more frequently to catch vertical layout changes
            const isMobileInit = this.options.width < 600 || (window.innerWidth && window.innerWidth < 600);
            setTimeout(checkResize, 50);
            setTimeout(checkResize, 200);
            setTimeout(checkResize, 500);
            if (isMobileInit) {
                // Additional checks for mobile to ensure vertical layout is calculated correctly
                setTimeout(checkResize, 800);
                setTimeout(checkResize, 1200);
                // Also check after first draw to ensure canvas is properly sized
                setTimeout(() => {
                    if (this.canvas && this.isRunning) {
                        // Force a draw and resize to ensure correct dimensions
                        this.draw();
                        checkResize();
                    }
                }, 1500);
            }
            
            // Also listen to container visibility changes
            if (typeof MutationObserver !== 'undefined') {
                this.mutationObserver = new MutationObserver(() => {
                    checkResize();
                });
                this.mutationObserver.observe(this.element, {
                    attributes: true,
                    attributeFilter: ['style', 'class']
                });
            }
        }

        /**
         * Create and setup canvas element
         */
        createCanvas() {
            // Create canvas
            this.canvas = document.createElement('canvas');
            this.canvas.className = 'countdown-timer-pro-canvas';
            this.canvas.width = this.options.width;
            this.canvas.height = this.options.height;
            
            // Get context
            this.ctx = this.canvas.getContext('2d');
            
            // Enable high-quality antialiasing for smooth circles
            this.ctx.imageSmoothingEnabled = true;
            this.ctx.imageSmoothingQuality = 'high';
            
            // Clear element
            this.$element.empty();
            
            // Append canvas first
            this.$element.append(this.canvas);
            
            // Set canvas size
            this.resize();
        }

        /**
         * Resize canvas to maintain aspect ratio
         */
        resize() {
            const container = this.$element;
            
            // Get actual container dimensions including padding
            const containerElement = container[0];
            if (!containerElement || !containerElement.offsetWidth) {
                return; // Container not ready
            }
            
            const containerWidth = containerElement.offsetWidth || container.width() || this.options.width;
            const containerHeight = containerElement.offsetHeight || container.height() || this.options.height;
            
            // Get computed styles to account for padding
            const computedStyle = window.getComputedStyle(containerElement);
            const paddingLeft = parseFloat(computedStyle.paddingLeft) || 0;
            const paddingRight = parseFloat(computedStyle.paddingRight) || 0;
            const paddingTop = parseFloat(computedStyle.paddingTop) || 0;
            const paddingBottom = parseFloat(computedStyle.paddingBottom) || 0;
            
            const availableWidth = containerWidth - paddingLeft - paddingRight;
            const availableHeight = containerHeight - paddingTop - paddingBottom;
            
            // Check if mobile layout (width < 600px)
            const isMobile = this.options.width < 600 || availableWidth < 600;
            
            // Maintain aspect ratio, but respect original width option for mobile layout
            const aspectRatio = this.options.width / this.options.height;
            let newWidth = availableWidth;
            let newHeight = availableWidth / aspectRatio;
            
            // If original width is < 600px (mobile), maintain that width for vertical layout
            // Don't scale up beyond original width to preserve mobile layout
            if (isMobile) {
                newWidth = Math.min(availableWidth, this.options.width);
                // For mobile vertical layout, calculate height based on content
                // Estimate: 4 circles (days, hours, minutes, seconds) with spacing
                const estimatedCircleRadius = Math.min(newWidth * 0.375, 50); // Max 75% of width / 2, cap at 50px
                const estimatedSpacing = Math.max(8, estimatedCircleRadius * 0.2); // Tighter spacing
                const totalVisible = (this.options.showDays ? 1 : 0) + 
                                    (this.options.showHours ? 1 : 0) + 
                                    (this.options.showMinutes ? 1 : 0) + 
                                    (this.options.showSeconds ? 1 : 0) || 4;
                const mobilePadding = Math.max(10, Math.min(20, newWidth * 0.02)); // 2% padding
                const estimatedTotalHeight = (estimatedCircleRadius * 2 * totalVisible) + 
                                           (estimatedSpacing * (totalVisible - 1)) + 
                                           (mobilePadding * 2);
                // Use estimated height - ensure it's tall enough for vertical layout
                // Don't use aspect ratio for mobile - use calculated height
                newHeight = Math.max(estimatedTotalHeight, this.options.height * 1.5); // At least 1.5x original for vertical
            } else {
                // If height is the limiting factor
                if (newHeight > availableHeight) {
                    newHeight = availableHeight;
                    newWidth = availableHeight * aspectRatio;
                }
                
                // Ensure minimum size (responsive minimums) - but not for mobile
                if (this.options.width >= 600) {
                    const minWidth = Math.max(250, containerWidth * 0.8);
                    const minHeight = Math.max(180, containerHeight * 0.6);
                    newWidth = Math.max(newWidth, minWidth);
                    newHeight = Math.max(newHeight, minHeight);
                }
            }
            
            // Check if size actually changed to avoid unnecessary updates
            const currentWidth = parseFloat(this.canvas.style.width) || this.canvas.width / (window.devicePixelRatio || 1);
            const currentHeight = parseFloat(this.canvas.style.height) || this.canvas.height / (window.devicePixelRatio || 1);
            
            // On mobile, always update if height changes significantly (more than 5px) to allow expansion
            // Don't skip updates on mobile - we need the canvas to expand vertically
            const isMobileResize = this.options.width < 600 || availableWidth < 600;
            if (isMobileResize) {
                // On mobile, always update if height is significantly different (allow expansion)
                if (Math.abs(currentWidth - newWidth) < 2 && Math.abs(currentHeight - newHeight) < 5) {
                    // Still allow updates if height needs to increase
                    if (newHeight <= currentHeight) {
                        return;
                    }
                }
            } else {
                // Desktop: only update if change is significant (more than 2px) to prevent jitter
                if (Math.abs(currentWidth - newWidth) < 2 && Math.abs(currentHeight - newHeight) < 2) {
                    return;
                }
            }
            
            // Use requestAnimationFrame to prevent layout thrashing
            requestAnimationFrame(() => {
                this.canvas.style.width = newWidth + 'px';
                this.canvas.style.height = newHeight + 'px';
                
                // Set actual canvas size (for retina displays)
                const dpr = window.devicePixelRatio || 1;
                this.canvas.width = newWidth * dpr;
                this.canvas.height = newHeight * dpr;
                
                // Reset transform and scale
                this.ctx.setTransform(1, 0, 0, 1, 0, 0);
                this.ctx.scale(dpr, dpr);
                
                // Re-enable high-quality antialiasing after resize
                this.ctx.imageSmoothingEnabled = true;
                this.ctx.imageSmoothingQuality = 'high';
                
                // Update celebration canvas size if it exists
                if (this.celebrationCanvas) {
                    this.celebrationCanvas.width = newWidth * dpr;
                    this.celebrationCanvas.height = newHeight * dpr;
                    this.celebrationCanvas.style.width = newWidth + 'px';
                    this.celebrationCanvas.style.height = newHeight + 'px';
                    this.celebrationCtx.setTransform(1, 0, 0, 1, 0, 0);
                    this.celebrationCtx.scale(dpr, dpr);
                    // Clear celebration canvas on resize
                    this.celebrationCtx.clearRect(0, 0, newWidth, newHeight);
                }
                
                // Redraw
                this.draw();
            });
        }

        /**
         * Calculate remaining/elapsed time based on timer type
         * 
         * @description
         * Calculates the current time state for countdown or countup modes.
         * For countdown: calculates remaining time until endTime.
         * For countup: calculates elapsed time since startTime.
         * Rounds endTime to nearest second to ensure exact duration display
         * (e.g., 3600 seconds shows as exactly 60:00, not 59:59).
         */
        calculateTime() {
            const now = Date.now();
            
            // Ensure remainingTime object exists
            if (!this.remainingTime) {
                this.remainingTime = {
                    days: 0,
                    hours: 0,
                    minutes: 0,
                    seconds: 0,
                    total: 0
                };
            }
            
            if (this.options.timeType === 'countdown') {
                // Countdown mode
                if (this.options.timeSource === 'server') {
                    // Server-side timestamp (Unix timestamp in seconds)
                    this.endTime = this.options.endTime * 1000;
                    this.remainingTime.total = Math.max(0, this.endTime - now);
                } else {
                    // Client-side duration
                    if (!this.startTime) {
                        this.startTime = now;
                        // Calculate exact endTime - round DOWN to prevent seconds from jumping up
                        // This ensures timer starts at correct value and only decreases
                        const exactDuration = (this.options.duration || 0) * 1000;
                        const targetEndTime = now + exactDuration;
                        // Round DOWN to prevent seconds from increasing at start
                        this.endTime = Math.floor(targetEndTime / 1000) * 1000;
                    }
                    this.remainingTime.total = Math.max(0, this.endTime - now);
                }
            } else {
                // Countup mode
                // Always reset startTime if it's null to ensure timer starts from 0
                if (this.startTime === null || this.startTime === undefined) {
                    // For countup, always start from current time (0 elapsed)
                    this.startTime = this.options.startTime ? 
                        (this.options.startTime * 1000) : now;
                }
                const elapsed = now - this.startTime;
                
                // If duration is set, limit countup to that maximum value
                if (this.options.duration && this.options.duration > 0) {
                    const maxDuration = this.options.duration * 1000;
                    // Clamp the elapsed time to the maximum duration
                    this.remainingTime.total = Math.min(Math.max(0, elapsed), maxDuration);
                } else {
                    this.remainingTime.total = Math.max(0, elapsed);
                }
            }
            
            // Ensure total is a valid number
            if (isNaN(this.remainingTime.total) || !isFinite(this.remainingTime.total)) {
                this.remainingTime.total = 0;
            }
            
            // Calculate time units
            const total = Math.max(0, Math.floor(this.remainingTime.total));
            this.remainingTime.days = Math.floor(total / (1000 * 60 * 60 * 24));
            this.remainingTime.hours = Math.floor((total % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
            this.remainingTime.minutes = Math.floor((total % (1000 * 60 * 60)) / (1000 * 60));
            this.remainingTime.seconds = Math.floor((total % (1000 * 60)) / 1000);
        }

        /**
         * Start the timer
         */
        start() {
            if (this.isRunning) return;
            
            this.isRunning = true;
            this.animate();
        }

        /**
         * Stop the timer
         */
        stop() {
            this.isRunning = false;
            if (this.animationFrame) {
                cancelAnimationFrame(this.animationFrame);
            }
        }

        /**
         * Reset the timer
         */
        reset() {
            this.stop();
            this.startTime = null;
            this.endTime = null;
            this.animationProgress = 0;
            this.colorIndex = 0;
            this.calculateTime();
            this.start();
        }

        /**
         * Animation loop
         */
        animate() {
            if (!this.isRunning) return;
            
            // Check if element is still in DOM and visible - CRITICAL for performance
            if (!this.element || !document.body.contains(this.element)) {
                this.stop();
                return;
            }
            
            // Check if canvas is still valid
            if (!this.canvas || !this.ctx) {
                this.stop();
                return;
            }
            
            this.calculateTime();
            
            // Update 3D rotation
            if (this.options.effects3D.enabled && this.options.effects3D.rotationSpeed > 0) {
                this.rotation3D += this.options.effects3D.rotationSpeed;
                if (this.rotation3D >= Math.PI * 2) {
                    this.rotation3D = 0;
                }
            }
            
            // Play tick sound if enabled (only once per second)
            if (this.options.sound && this.options.sound.enabled && this.options.sound.tickEnabled) {
                const currentSecond = Math.floor(this.remainingTime.total / 1000);
                if (currentSecond !== this.lastSecondPlayed && currentSecond > 0) {
                    this.playTickSound();
                    this.lastSecondPlayed = currentSecond;
                }
            }
            
            // Only draw if canvas exists and is valid
            this.draw();
            
            // Check if timer finished
            if (this.options.timeType === 'countdown' && this.remainingTime.total <= 0) {
                this.onFinish();
                return;
            }
            
            // Check if countup reached maximum duration
            if (this.options.timeType === 'countup' && this.options.duration && this.options.duration > 0) {
                const maxDuration = this.options.duration * 1000;
                // Check if we've reached or exceeded the maximum duration
                if (this.remainingTime.total >= maxDuration) {
                    // Clamp to exact maximum to prevent exceeding
                    this.remainingTime.total = maxDuration;
                    // Stop the timer
                    this.stop();
                    // Trigger finish callback
                    this.onFinish();
                    return;
                }
            }
            
            // Continue animation
            this.animationFrame = requestAnimationFrame(() => this.animate());
        }
        
        /**
         * Initialize audio context for sound effects
         */
        initAudio() {
            try {
                const AudioContext = window.AudioContext || window.webkitAudioContext;
                if (AudioContext) {
                    this.audioContext = new AudioContext();
                    this.soundEnabled = true;
                }
            } catch (e) {
                // Audio context not supported - silently disable sound
                this.soundEnabled = false;
            }
        }
        
        /**
         * Play tick sound (every second)
         */
        playTickSound() {
            if (!this.soundEnabled || !this.audioContext) return;
            
            try {
                const oscillator = this.audioContext.createOscillator();
                const gainNode = this.audioContext.createGain();
                
                oscillator.connect(gainNode);
                gainNode.connect(this.audioContext.destination);
                
                oscillator.frequency.value = this.options.sound.tickFrequency;
                oscillator.type = 'sine';
                
                gainNode.gain.setValueAtTime(0, this.audioContext.currentTime);
                gainNode.gain.linearRampToValueAtTime(this.options.sound.tickVolume, this.audioContext.currentTime + 0.01);
                gainNode.gain.linearRampToValueAtTime(0, this.audioContext.currentTime + 0.1);
                
                oscillator.start(this.audioContext.currentTime);
                oscillator.stop(this.audioContext.currentTime + 0.1);
            } catch (e) {
                // Silently fail if sound cannot be played
            }
        }
        
        /**
         * Play finish sound
         */
        playFinishSound() {
            if (!this.soundEnabled || !this.audioContext) return;
            
            try {
                const oscillator = this.audioContext.createOscillator();
                const gainNode = this.audioContext.createGain();
                
                oscillator.connect(gainNode);
                gainNode.connect(this.audioContext.destination);
                
                oscillator.frequency.setValueAtTime(this.options.sound.finishFrequency, this.audioContext.currentTime);
                oscillator.frequency.exponentialRampToValueAtTime(this.options.sound.finishFrequency * 0.5, this.audioContext.currentTime + 0.5);
                oscillator.type = 'sine';
                
                gainNode.gain.setValueAtTime(0, this.audioContext.currentTime);
                gainNode.gain.linearRampToValueAtTime(this.options.sound.finishVolume, this.audioContext.currentTime + 0.05);
                gainNode.gain.linearRampToValueAtTime(0, this.audioContext.currentTime + 0.5);
                
                oscillator.start(this.audioContext.currentTime);
                oscillator.stop(this.audioContext.currentTime + 0.5);
            } catch (e) {
                // Silently fail if sound cannot be played
            }
        }
        
        /**
         * Initialize real-time synchronization system
         * FEATURE 1: Cross-tab synchronization with optional WebSocket server support for multi-user sync
         * 
         * @description
         * Sets up WebSocket connection (if server URL provided) or uses localStorage
         * fallback for cross-tab/window synchronization. All instances in the same
         * room will have synchronized timers and viewer counts.
         */
        initRealTimeSync() {
            const sync = this.options.realTimeSync;
            if (!sync || !sync.enabled) return;
            
            // Generate or use provided room ID
            // If roomId is provided, use it exactly as provided
            // If not provided, generate a unique one
            if (sync.roomId) {
                this.realTimeSync.roomId = String(sync.roomId).trim();
            } else {
                this.realTimeSync.roomId = this.generateRoomId();
            }
            this.realTimeSync.enabled = true;
            
            // Immediately write this instance to localStorage for other instances to detect
            this.updateLocalStorageSync();
            
            // Try WebSocket connection if server URL provided
            if (sync.serverUrl) {
                this.connectWebSocket();
            } else if (sync.useLocalStorageFallback) {
                // Use localStorage fallback for synchronization
                this.initLocalStorageSync();
            }
            
            // Create UI elements for viewer count and connection status
            this.createRealTimeSyncUI();
            
            // Initial viewer count update
            this.updateViewerCount();
            
            // Start sync interval
            this.startSyncInterval();
        }
        
        /**
         * Generate unique room ID for real-time synchronization
         * 
         * @returns {string} Unique room ID in format 'room_xxxxx_timestamp'
         * @description
         * Creates a unique identifier for the synchronization room. Used to group
         * multiple timer instances that should be synchronized together. Combines
         * random string with timestamp for uniqueness.
         */
        generateRoomId() {
            return 'room_' + Math.random().toString(36).substr(2, 9) + '_' + Date.now().toString(36);
        }
        
        /**
         * Connect to WebSocket server for real-time synchronization
         * 
         * @description
         * Establishes WebSocket connection to server for real-time timer synchronization.
         * Falls back to localStorage if connection fails. Handles reconnection
         * automatically with exponential backoff.
         */
        connectWebSocket() {
            const sync = this.options.realTimeSync;
            if (!sync.serverUrl) return;
            
            try {
                this.realTimeSync.connectionStatus = 'connecting';
                
                const wsUrl = sync.serverUrl + '?roomId=' + encodeURIComponent(this.realTimeSync.roomId);
                this.realTimeSync.ws = new WebSocket(wsUrl);
                this.realTimeSync.wsUrl = sync.serverUrl;
                
                this.realTimeSync.ws.onopen = () => {
                    this.realTimeSync.connectionStatus = 'connected';
                    this.realTimeSync.reconnectAttempts = 0;
                    this.sendSyncData();
                };
                
                this.realTimeSync.ws.onmessage = (event) => {
                    try {
                        const data = JSON.parse(event.data);
                        this.handleSyncMessage(data);
                    } catch (e) {
                        console.error('Error parsing sync message:', e);
                    }
                };
                
                this.realTimeSync.ws.onerror = (error) => {
                    console.error('WebSocket error:', error);
                    this.realTimeSync.connectionStatus = 'disconnected';
                };
                
                this.realTimeSync.ws.onclose = () => {
                    this.realTimeSync.connectionStatus = 'disconnected';
                    this.attemptReconnect();
                };
            } catch (e) {
                console.error('Error connecting WebSocket:', e);
                this.realTimeSync.connectionStatus = 'disconnected';
                
                // Fallback to localStorage if WebSocket fails
                if (sync.useLocalStorageFallback) {
                    this.initLocalStorageSync();
                }
            }
        }
        
        /**
         * Initialize localStorage-based synchronization as fallback
         * 
         * @description
         * Sets up localStorage-based synchronization when WebSocket is unavailable.
         * Uses storage events to detect updates from other tabs/windows and polling
         * to actively check for changes. This ensures synchronization works even
         * without a WebSocket server.
         */
        initLocalStorageSync() {
            const storageKey = 'countdown_timer_sync_' + this.realTimeSync.roomId;
            
            // Register this instance
            this.updateLocalStorageSync();
            
            // Listen for storage events from other tabs/windows
            const roomKey = 'countdown_timer_sync_' + this.realTimeSync.roomId;
            window.addEventListener('storage', (e) => {
                // Listen for any instance updates in this room
                if (e.key && e.key.startsWith(roomKey + '_')) {
                    try {
                        const data = JSON.parse(e.newValue);
                        if (data && data.roomId === this.realTimeSync.roomId && data.instanceId !== this.instanceId) {
                            this.handleSyncMessage(data);
                            // Update viewer count when other instances update
                            this.updateViewerCount();
                        }
                    } catch (err) {
                        // Ignore parse errors
                    }
                }
            });
            
            this.realTimeSync.connectionStatus = 'connected';
        }
        
        /**
         * Update localStorage with current sync data
         * 
         * @description
         * Stores this instance's timer state in localStorage so other instances
         * in the same room can detect and synchronize with it. Removes any old
         * entries for this instance to prevent duplicates.
         */
        updateLocalStorageSync() {
            if (!this.realTimeSync.enabled || !this.realTimeSync.roomId) return;
            
            const roomKey = 'countdown_timer_sync_' + this.realTimeSync.roomId;
            const instanceKey = roomKey + '_' + this.instanceId;
            
            // First, remove any old entries for this instance (to prevent duplicates)
            try {
                for (let i = localStorage.length - 1; i >= 0; i--) {
                    const key = localStorage.key(i);
                    if (key && key.startsWith(roomKey + '_') && key !== instanceKey) {
                        try {
                            const data = JSON.parse(localStorage.getItem(key));
                            if (data && data.instanceId === this.instanceId) {
                                // Found old entry for this instance, remove it
                                localStorage.removeItem(key);
                            }
                        } catch (e) {
                            // Ignore parse errors
                        }
                    }
                }
            } catch (e) {
                // Ignore errors
            }
            
            // Store this instance's data
            const instanceData = {
                roomId: this.realTimeSync.roomId,
                timestamp: Date.now(),
                remainingTime: this.remainingTime.total,
                instanceId: this.instanceId
            };
            
            try {
                // Store this instance's data with unique key
                localStorage.setItem(instanceKey, JSON.stringify(instanceData));
                
            } catch (e) {
                console.error('[CountdownTimerPro] Error writing to localStorage:', e);
            }
        }
        
        /**
         * Start periodic sync interval
         * 
         * @description
         * Sets up a periodic interval to update localStorage and check for other
         * instances. Runs at configurable interval (default: 500ms) to balance
         * sync frequency with performance. Updates viewer count on each cycle.
         */
        startSyncInterval() {
            if (this.realTimeSync.syncInterval) {
                clearInterval(this.realTimeSync.syncInterval);
            }
            
            // Send sync data and update viewer count
            // Use 500ms interval for balance between sync frequency and performance
            const syncInterval = Math.min(500, this.options.realTimeSync.syncInterval);
            this.realTimeSync.syncInterval = setInterval(() => {
                // Always update localStorage first (this ensures we're always visible)
                this.updateLocalStorageSync();
                // Check for other instances' updates (polling)
                this.checkOtherInstances();
                // Update viewer count
                this.updateViewerCount();
            }, syncInterval);
        }
        
        /**
         * Check for updates from other instances using polling method
         * 
         * @description
         * This method actively checks localStorage for updates from other instances.
         * It's needed because storage events don't fire in the same window/tab.
         * When another instance is found with a more recent timestamp, the timer
         * synchronizes to match it.
         */
        checkOtherInstances() {
            if (!this.realTimeSync.enabled || !this.realTimeSync.roomId) return;
            
            const roomKey = 'countdown_timer_sync_' + this.realTimeSync.roomId;
            const now = Date.now();
            const activeThreshold = 10000; // 10 seconds (increased to prevent premature removal)
            
            try {
                let foundOtherInstance = false;
                let mostRecentTime = null;
                let mostRecentInstance = null;
                let allInstances = [];
                let allKeys = [];
                
                // Check all localStorage keys
                for (let i = 0; i < localStorage.length; i++) {
                    const key = localStorage.key(i);
                    if (key && key.startsWith('countdown_timer_sync_')) {
                        allKeys.push(key);
                        if (key.startsWith(roomKey + '_')) {
                            try {
                                const instanceData = JSON.parse(localStorage.getItem(key));
                                if (instanceData && instanceData.instanceId) {
                                    allInstances.push({
                                        instanceId: instanceData.instanceId,
                                        timestamp: instanceData.timestamp,
                                        age: now - (instanceData.timestamp || 0),
                                        roomId: instanceData.roomId
                                    });
                                    
                                    if (instanceData.instanceId !== this.instanceId) {
                                        // Check if instance is active
                                        if (instanceData.timestamp && (now - instanceData.timestamp) < activeThreshold) {
                                            foundOtherInstance = true;
                                            
                                            // Find the most recent instance (most up-to-date timer)
                                            if (!mostRecentTime || instanceData.timestamp > mostRecentTime) {
                                                mostRecentTime = instanceData.timestamp;
                                                mostRecentInstance = instanceData;
                                            }
                                        }
                                    }
                                }
                            } catch (e) {
                                // Ignore invalid data
                            }
                        }
                    }
                }
                
                
                // Sync with the most recent instance if found
                // CRITICAL: Only sync if timer is actually running and not frozen
                if (foundOtherInstance && mostRecentInstance && mostRecentInstance.remainingTime !== undefined) {
                    const timeDiff = Math.abs(this.remainingTime.total - mostRecentInstance.remainingTime);
                    
                    // CRITICAL: Don't sync if timer appears frozen (same value for too long)
                    // Check if our timer is actually updating (not stuck)
                    const lastRemainingTime = this.realTimeSync.lastRemainingTime || this.remainingTime.total;
                    const timeSinceLastChange = Math.abs(this.remainingTime.total - lastRemainingTime);
                    
                    // Only sync if:
                    // 1. Difference is significant (> 500ms) to avoid sync loops
                    // 2. Our timer is actually updating (not frozen)
                    // 3. We haven't synced too recently (prevent loops)
                    if (timeDiff > 500 && timeSinceLastChange > 100) {
                        const lastSyncTime = this.realTimeSync.lastSyncTime || 0;
                        if (now - lastSyncTime > 3000) { // Only sync every 3 seconds max (prevent blocking)
                            this.realTimeSync.lastSyncTime = now;
                            this.realTimeSync.lastRemainingTime = this.remainingTime.total;
                            this.syncTimerFromServer(mostRecentInstance.remainingTime);
                        }
                    }
                }
            } catch (e) {
                console.error('[CountdownTimerPro] Error checking instances:', e);
            }
        }
        
        /**
         * Send sync data to server or localStorage
         * 
         * @description
         * Broadcasts this instance's current timer state to other instances.
         * Updates localStorage for cross-tab/window sync and sends via WebSocket
         * if connected. Called periodically to keep all instances synchronized.
         */
        sendSyncData() {
            if (!this.realTimeSync.enabled) return;
            
            // Always update localStorage (used by polling)
            this.updateLocalStorageSync();
            
            // Send via WebSocket if connected
            if (this.realTimeSync.ws && this.realTimeSync.ws.readyState === WebSocket.OPEN) {
                try {
                    const syncData = {
                        roomId: this.realTimeSync.roomId,
                        timestamp: Date.now(),
                        remainingTime: this.remainingTime.total,
                        instanceId: this.instanceId
                    };
                    this.realTimeSync.ws.send(JSON.stringify({
                        type: 'sync',
                        data: syncData
                    }));
                } catch (e) {
                    console.error('Error sending WebSocket message:', e);
                }
            }
        }
        
        /**
         * Handle sync message from server or localStorage
         * 
         * @param {Object} data - Sync data object with roomId, instanceId, remainingTime, timestamp
         * @description
         * Processes sync messages from other instances. Updates viewer count
         * and synchronizes timer if the other instance has a more recent value.
         * Prevents syncing with self to avoid loops.
         */
        handleSyncMessage(data) {
            if (!data || data.roomId !== this.realTimeSync.roomId) return;
            
            // Update viewer count by counting active instances
            const oldCount = this.realTimeSync.viewerCount;
            this.updateViewerCount();
            if (oldCount !== this.realTimeSync.viewerCount) {
                this.triggerEvent('viewerCountChanged', this.realTimeSync.viewerCount);
            }
            
            // Sync timer if data is from another instance (not this one)
            if (data.instanceId && data.instanceId !== this.instanceId && data.remainingTime !== undefined) {
                const timeDiff = Math.abs(this.remainingTime.total - data.remainingTime);
                // Sync if difference is significant (> 500ms) - reduced threshold for better sync
                if (timeDiff > 500) {
                    this.syncTimerFromServer(data.remainingTime);
                }
            }
            
            this.realTimeSync.lastSyncTime = Date.now();
            this.triggerEvent('synced', data);
        }
        
        /**
         * Sync timer from server or other instance data
         * 
         * @param {number} serverRemainingTime - Remaining time in milliseconds from server/other instance
         * @description
         * Synchronizes this timer's endTime with the provided remaining time.
         * Used for real-time synchronization when another instance has a more
         * recent timer value. Prevents timer drift between multiple users.
         */
        syncTimerFromServer(serverRemainingTime) {
            // CRITICAL: Don't sync if timer appears frozen - this prevents blocking
            const currentRemaining = this.remainingTime.total;
            const lastRemaining = this.realTimeSync.lastRemainingTime || currentRemaining;
            
            // If timer hasn't changed in last 2 seconds, it might be frozen - don't sync
            if (Math.abs(currentRemaining - lastRemaining) < 100 && currentRemaining > 0) {
                // Timer might be frozen - skip sync to prevent blocking
                return;
            }
            
            if (this.options.timeType === 'countdown') {
                // Only sync if difference is significant (> 1000ms) to avoid jitter and blocking
                const timeDiff = Math.abs(this.remainingTime.total - serverRemainingTime);
                if (timeDiff > 1000) {
                    // Always sync endTime to match the server/other instance
                    const newEndTime = Date.now() + serverRemainingTime;
                    this.endTime = newEndTime;
                    
                    // Reset startTime to ensure proper calculation
                    if (this.options.timeSource === 'client') {
                        this.startTime = Date.now() - (this.options.duration * 1000 - serverRemainingTime);
                    }
                    
                    // Update last remaining time to track if timer is frozen
                    this.realTimeSync.lastRemainingTime = this.remainingTime.total;
                }
            }
            // Recalculate time immediately
            this.calculateTime();
        }
        
        /**
         * Update viewer count from localStorage or server
         */
        updateViewerCount() {
            if (!this.realTimeSync.enabled) return;
            
            const roomKey = 'countdown_timer_sync_' + this.realTimeSync.roomId;
            const now = Date.now();
            const activeThreshold = 10000; // 10 seconds (increased to prevent premature removal)
            
            try {
                // First, ensure this instance is written to localStorage
                this.updateLocalStorageSync();
                
                // Count all active instances for this room
                let activeCount = 0;
                const activeInstances = new Set();
                const allFoundInstances = [];
                
                // Use a more lenient threshold for counting (15 seconds)
                const countThreshold = 15000;
                
                // Iterate through all localStorage keys
                for (let i = 0; i < localStorage.length; i++) {
                    const key = localStorage.key(i);
                    if (key && key.startsWith(roomKey + '_')) {
                        try {
                            const instanceData = JSON.parse(localStorage.getItem(key));
                            if (instanceData && instanceData.instanceId) {
                                const timeSinceUpdate = now - (instanceData.timestamp || 0);
                                
                                allFoundInstances.push({
                                    instanceId: instanceData.instanceId,
                                    timestamp: instanceData.timestamp,
                                    age: timeSinceUpdate,
                                    isActive: timeSinceUpdate < countThreshold,
                                    isCurrentInstance: instanceData.instanceId === this.instanceId
                                });
                                
                                // Check if instance is active (timestamp within threshold)
                                if (instanceData.timestamp && timeSinceUpdate < countThreshold) {
                                    // Count unique instances only (including current instance)
                                    if (!activeInstances.has(instanceData.instanceId)) {
                                        activeInstances.add(instanceData.instanceId);
                                        activeCount++;
                                    }
                                } else if (timeSinceUpdate >= activeThreshold) {
                                    // Don't remove stale instance data immediately - wait much longer
                                    // Only remove if really stale (> 30 seconds) to prevent premature removal
                                    // Never remove current instance
                                    if (instanceData.instanceId !== this.instanceId && timeSinceUpdate > 30000) {
                                        localStorage.removeItem(key);
                                    }
                                }
                            }
                        } catch (e) {
                            // Ignore invalid data
                        }
                    }
                }
                
                // CRITICAL: Count current instance first to prevent double counting
                // Only add if not already in the set (prevents counting same instance twice)
                if (!activeInstances.has(this.instanceId)) {
                    activeInstances.add(this.instanceId);
                    activeCount++;
                }
                
                // Ensure at least 1 (this instance) - but don't double count
                activeCount = Math.max(1, activeCount);
                
                // CRITICAL: Limit max count to prevent showing 25+ viewers from same browser
                // If count is suspiciously high (> 10), it's likely a bug - cap it
                const newCount = Math.min(Math.max(1, activeCount), 10);
                
                // Update viewer count only if changed (prevents unnecessary UI updates)
                if (this.realTimeSync.viewerCount !== newCount) {
                    this.realTimeSync.viewerCount = newCount;
                    this.updateViewerCountUI();
                }
            } catch (e) {
                console.error('[CountdownTimerPro] Error updating viewer count:', e);
                // Fallback: at least count this instance
                this.realTimeSync.viewerCount = 1;
                this.updateViewerCountUI();
            }
        }
        
        /**
         * Attempt to reconnect WebSocket with exponential backoff
         * 
         * @description
         * Handles automatic reconnection when WebSocket connection is lost.
         * Respects maxReconnectAttempts limit and falls back to localStorage
         * if max attempts are reached. Uses configurable reconnectInterval
         * for delay between attempts.
         */
        attemptReconnect() {
            const sync = this.options.realTimeSync;
            if (this.realTimeSync.reconnectAttempts >= sync.maxReconnectAttempts) {
                // Max attempts reached, fallback to localStorage
                if (sync.useLocalStorageFallback) {
                    this.initLocalStorageSync();
                }
                return;
            }
            
            this.realTimeSync.reconnectAttempts++;
            setTimeout(() => {
                this.connectWebSocket();
            }, sync.reconnectInterval);
        }
        
        /**
         * Create UI elements for real-time sync features
         * 
         * @description
         * Creates and displays UI elements for real-time synchronization features
         * including viewer count display. Uses fixed styling that doesn't change
         * with timer presets to maintain consistent appearance.
         */
        createRealTimeSyncUI() {
            if (!this.options.realTimeSync.showViewerCount) {
                return;
            }
            
            const uiContainer = document.createElement('div');
            uiContainer.className = 'countdown-timer-pro-realtime-ui';
            // Check if mobile (width < 600px)
            const isMobile = (this.options.width && this.options.width < 600) || 
                            (this.$element.width() && this.$element.width() < 600) ||
                            (window.innerWidth && window.innerWidth < 600);
            
            if (isMobile) {
                // Mobile: center top
                uiContainer.style.cssText = `
                    position: absolute;
                    top: 10px;
                    left: 50%;
                    transform: translateX(-50%);
                    z-index: 1000;
                    display: flex;
                    flex-direction: column;
                    gap: 8px;
                `;
            } else {
                // Desktop: top right
                uiContainer.style.cssText = `
                    position: absolute;
                    top: 10px;
                    right: 10px;
                    z-index: 1000;
                    display: flex;
                    flex-direction: column;
                    gap: 8px;
                `;
            }
            
            // Viewer count display - Fixed style (doesn't change with presets)
            if (this.options.realTimeSync.showViewerCount) {
                const viewerCountEl = document.createElement('div');
                viewerCountEl.className = 'countdown-timer-pro-viewer-count';
                viewerCountEl.style.cssText = `
                    background: rgba(255, 255, 255, 0.95);
                    padding: 6px 12px;
                    border-radius: 20px;
                    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.15);
                    display: flex;
                    align-items: center;
                    gap: 6px;
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
                    font-size: 12px;
                    font-weight: 600;
                    color: #2c3e50;
                    border: 1px solid rgba(0, 0, 0, 0.05);
                `;
                // Use textContent and createElement instead of innerHTML for CSP compliance
                const iconSpan = document.createElement('span');
                iconSpan.style.fontSize = '14px';
                iconSpan.textContent = '👥';
                
                const countSpan = document.createElement('span');
                countSpan.className = 'viewer-count-number';
                countSpan.style.fontWeight = '700';
                countSpan.style.color = '#667eea';
                countSpan.textContent = '1';
                
                const textSpan = document.createElement('span');
                textSpan.style.color = '#555';
                textSpan.textContent = ' people watching';
                
                viewerCountEl.appendChild(iconSpan);
                viewerCountEl.appendChild(countSpan);
                viewerCountEl.appendChild(textSpan);
                this.realTimeSync.viewerCountElement = viewerCountEl;
                uiContainer.appendChild(viewerCountEl);
            }
            
            this.$element.append(uiContainer);
            this.realTimeSync.uiContainer = uiContainer;
            
            // Initial update
            this.updateViewerCountUI();
        }
        
        /**
         * Update viewer count UI
         */
        updateViewerCountUI() {
            if (!this.realTimeSync.viewerCountElement || !this.realTimeSync.uiContainer) return;
            
            // Update position for mobile if needed
            const isMobile = (this.options.width && this.options.width < 600) || 
                            (this.$element.width() && this.$element.width() < 600) ||
                            (window.innerWidth && window.innerWidth < 600);
            
            if (isMobile) {
                // Mobile: center top
                this.realTimeSync.uiContainer.style.top = '10px';
                this.realTimeSync.uiContainer.style.left = '50%';
                this.realTimeSync.uiContainer.style.right = 'auto';
                this.realTimeSync.uiContainer.style.transform = 'translateX(-50%)';
            } else {
                // Desktop: top right
                this.realTimeSync.uiContainer.style.top = '10px';
                this.realTimeSync.uiContainer.style.right = '10px';
                this.realTimeSync.uiContainer.style.left = 'auto';
                this.realTimeSync.uiContainer.style.transform = 'none';
            }
            
            const countEl = this.realTimeSync.viewerCountElement.querySelector('.viewer-count-number');
            if (countEl) {
                const oldCount = parseInt(countEl.textContent) || 1;
                const newCount = this.realTimeSync.viewerCount || 1;
                countEl.textContent = newCount;
                
                // Animate change
                if (oldCount !== newCount) {
                    countEl.style.transform = 'scale(1.2)';
                    setTimeout(() => {
                        countEl.style.transform = 'scale(1)';
                    }, 300);
                }
            }
        }
        
        
        /**
         * Trigger custom event
         */
        triggerEvent(eventName, data) {
            const event = new CustomEvent('countdownTimerPro:' + eventName, {
                detail: data,
                bubbles: true
            });
            this.element.dispatchEvent(event);
        }
        
        /**
         * Get current viewer count
         */
        getViewerCount() {
            return this.realTimeSync.viewerCount || 0;
        }
        
        /**
         * Get connection status
         */
        getConnectionStatus() {
            return this.realTimeSync.connectionStatus || 'disconnected';
        }
        
        /**
         * Manually sync with server
         */
        sync() {
            this.sendSyncData();
        }
        
        /**
         * Disconnect from server
         */
        disconnect() {
            if (this.realTimeSync.syncInterval) {
                clearInterval(this.realTimeSync.syncInterval);
                this.realTimeSync.syncInterval = null;
            }
            
            if (this.realTimeSync.ws) {
                this.realTimeSync.ws.close();
                this.realTimeSync.ws = null;
            }
            
            // Remove this instance's data from localStorage
            try {
                const roomKey = 'countdown_timer_sync_' + this.realTimeSync.roomId;
                const instanceKey = roomKey + '_' + this.instanceId;
                localStorage.removeItem(instanceKey);
            } catch (e) {
                // Ignore errors
            }
            
            this.realTimeSync.connectionStatus = 'disconnected';
        }
        
        /**
         * ============================================
         * BEHAVIORAL TARGETING ENGINE (FEATURE 2: UNIQUE)
         * ============================================
         * Timer che cambia in base a comportamento utente
         */
        
        /**
         * Initialize behavioral targeting system
         * FEATURE 2: UNIQUE - Timer adapts based on user behavior
         * 
         * @description
         * Detects user behavior (new/returning visitor, device type, traffic source)
         * and applies matching rules to customize the timer duration and appearance.
         * This personalization increases conversion rates by showing appropriate
         * urgency levels to different user segments.
         */
        initBehavioralTargeting() {
            const targeting = this.options.behavioralTargeting;
            if (!targeting || !targeting.enabled) return;
            
            // Copy configuration
            this.behavioralTargeting.enabled = true;
            this.behavioralTargeting.rules = targeting.rules || [];
            this.behavioralTargeting.priority = targeting.priority || ['newVisitor', 'returningVisitor', 'device', 'source'];
            this.behavioralTargeting.storageKey = targeting.storageKey || 'countdown_timer_visits';
            this.behavioralTargeting.visitThreshold = targeting.visitThreshold || 2;
            
            // Track this visit
            this.trackVisit();
            
            // Detect behavior and apply rule
            const behavior = this.detectBehavior();
            const rule = this.findMatchingRule(behavior);
            
            if (rule) {
                this.applyBehavioralRule(rule, behavior);
            }
        }
        
        /**
         * Track visit in localStorage for behavioral targeting
         * 
         * @description
         * Records this visit in localStorage to determine if user is new or returning.
         * Maintains a list of visit timestamps (last 30 visits) to prevent localStorage
         * bloat. Used by behavioral targeting to apply appropriate rules.
         */
        trackVisit() {
            try {
                const storageKey = this.behavioralTargeting.storageKey;
                const now = Date.now();
                const visitData = JSON.parse(localStorage.getItem(storageKey) || '{"visits": [], "count": 0}');
                
                // Add this visit
                visitData.visits.push(now);
                visitData.count = visitData.visits.length;
                visitData.lastVisit = now;
                
                // Keep only last 30 visits (to prevent localStorage bloat)
                if (visitData.visits.length > 30) {
                    visitData.visits = visitData.visits.slice(-30);
                }
                
                localStorage.setItem(storageKey, JSON.stringify(visitData));
                
                this.behavioralTargeting.visitCount = visitData.count;
                this.behavioralTargeting.lastVisit = now;
            } catch (e) {
                console.error('[CountdownTimerPro] Error tracking visit:', e);
            }
        }
        
        /**
         * Detect user behavior for targeting
         * 
         * @returns {Object} Behavior object with detected properties
         * @description
         * Analyzes user's visit history, device type, and traffic source to
         * determine which behavioral targeting rule should be applied.
         * 
         * @returns {Object} Behavior object containing:
         *   - isNewVisitor: boolean
         *   - isReturningVisitor: boolean
         *   - device: string ('mobile', 'tablet', 'desktop')
         *   - source: string ('socialMedia', 'email', 'direct', 'search', 'other')
         *   - visitCount: number
         */
        detectBehavior() {
            const behavior = {
                isNewVisitor: false,
                isReturningVisitor: false,
                device: null,
                source: null,
                visitCount: 0
            };
            
            // Get visit count
            try {
                const storageKey = this.behavioralTargeting.storageKey;
                const visitData = JSON.parse(localStorage.getItem(storageKey) || '{"visits": [], "count": 0}');
                behavior.visitCount = visitData.count || 0;
                behavior.isNewVisitor = behavior.visitCount <= 1;
                behavior.isReturningVisitor = behavior.visitCount >= this.behavioralTargeting.visitThreshold;
            } catch (e) {
                // Default to new visitor if error
                behavior.isNewVisitor = true;
            }
            
            // Detect device
            behavior.device = this.detectDevice();
            
            // Detect source
            behavior.source = this.detectSource();
            
            return behavior;
        }
        
        /**
         * Detect device type from user agent and screen width
         * 
         * @returns {string} Device type: 'mobile', 'tablet', or 'desktop'
         * @description
         * Analyzes user agent string and screen width to determine device type.
         * Used for behavioral targeting to show appropriate timer durations.
         */
        detectDevice() {
            const width = window.innerWidth;
            const userAgent = navigator.userAgent.toLowerCase();
            
            // Check for mobile devices
            if (/mobile|android|iphone|ipod|blackberry|iemobile|opera mini/i.test(userAgent)) {
                return 'mobile';
            }
            
            // Check for tablets
            if (/tablet|ipad|playbook|silk/i.test(userAgent) || (width >= 600 && width < 1024)) {
                return 'tablet';
            }
            
            // Default to desktop
            return 'desktop';
        }
        
        /**
         * Detect traffic source from referrer and URL parameters
         * 
         * @returns {string} Traffic source: 'socialMedia', 'email', 'direct', 'search', or 'other'
         * @description
         * Analyzes document.referrer and URL parameters (utm_source) to determine
         * where the user came from. Used for behavioral targeting to customize
         * timer urgency based on traffic source.
         */
        detectSource() {
            const referrer = document.referrer.toLowerCase();
            const urlParams = new URLSearchParams(window.location.search);
            
            // Check URL parameters first (utm_source, ref, etc.)
            const utmSource = urlParams.get('utm_source');
            if (utmSource) {
                if (['facebook', 'twitter', 'instagram', 'linkedin', 'pinterest', 'tiktok', 'youtube'].includes(utmSource.toLowerCase())) {
                    return 'socialMedia';
                }
                if (['email', 'mail', 'newsletter'].includes(utmSource.toLowerCase())) {
                    return 'email';
                }
            }
            
            // Check referrer
            if (!referrer || referrer === '') {
                return 'direct';
            }
            
            // Social media domains
            const socialDomains = ['facebook.com', 'twitter.com', 'instagram.com', 'linkedin.com', 'pinterest.com', 'tiktok.com', 'youtube.com', 'reddit.com', 'snapchat.com'];
            if (socialDomains.some(domain => referrer.includes(domain))) {
                return 'socialMedia';
            }
            
            // Email domains (common email clients)
            const emailDomains = ['mail.', 'email.', 'newsletter.', 'campaign.'];
            if (emailDomains.some(domain => referrer.includes(domain))) {
                return 'email';
            }
            
            // Search engines
            const searchEngines = ['google.', 'bing.', 'yahoo.', 'duckduckgo.', 'yandex.'];
            if (searchEngines.some(engine => referrer.includes(engine))) {
                return 'search';
            }
            
            return 'other';
        }
        
        /**
         * Find matching rule based on detected behavior
         * 
         * @param {Object} behavior - Detected behavior object
         * @returns {Object|null} Matching rule or null if no rule matches
         * @description
         * Evaluates rules in priority order (newVisitor → returningVisitor → device → source)
         * and returns the first matching rule. This ensures that more specific rules
         * (like newVisitor) take precedence over general ones (like device).
         */
        findMatchingRule(behavior) {
            const rules = this.behavioralTargeting.rules;
            const priority = this.behavioralTargeting.priority;
            
            // Check rules in priority order
            for (const priorityKey of priority) {
                for (const rule of rules) {
                    if (this.ruleMatches(rule, behavior, priorityKey)) {
                        return rule;
                    }
                }
            }
            
            return null;
        }
        
        /**
         * Check if a behavioral rule matches the detected behavior
         * 
         * @param {Object} rule - Rule object with condition property
         * @param {Object} behavior - Detected behavior object
         * @param {string} priorityKey - Current priority key being checked ('newVisitor', 'returningVisitor', 'device', 'source')
         * @returns {boolean} True if rule condition matches the behavior for the given priority key
         * @description
         * Evaluates whether a rule's condition matches the detected behavior based
         * on the current priority key. Handles special cases like 'mobile' matching
         * both 'mobile' and 'tablet' device types.
         */
        ruleMatches(rule, behavior, priorityKey) {
            if (!rule.condition) return false;
            
            // Check condition based on priority key
            switch (priorityKey) {
                case 'newVisitor':
                    if (rule.condition === 'newVisitor' && behavior.isNewVisitor) {
                        return true;
                    }
                    break;
                case 'returningVisitor':
                    if (rule.condition === 'returningVisitor' && behavior.isReturningVisitor) {
                        return true;
                    }
                    break;
                case 'device':
                    if (rule.condition === behavior.device || 
                        rule.condition === 'mobile' && (behavior.device === 'mobile' || behavior.device === 'tablet') ||
                        rule.condition === 'desktop' && behavior.device === 'desktop') {
                        return true;
                    }
                    break;
                case 'source':
                    if (rule.condition === behavior.source) {
                        return true;
                    }
                    break;
            }
            
            return false;
        }
        
        /**
         * Apply behavioral rule to timer
         * 
         * @param {Object} rule - Rule object with condition, duration, and optional colors
         * @param {Object} behavior - Detected behavior object
         * @description
         * Applies a behavioral targeting rule by updating timer duration and
         * optionally colors. Shows visual indicator in demo mode. Triggers
         * 'behavioralRuleApplied' event.
         */
        applyBehavioralRule(rule, behavior) {
            if (!rule) return;
            
            this.behavioralTargeting.appliedRule = rule;
            
            // Update duration if specified
            if (rule.duration !== undefined) {
                const newDuration = typeof rule.duration === 'number' ? rule.duration : parseInt(rule.duration);
                if (!isNaN(newDuration) && newDuration > 0) {
                    // Update options duration
                    const oldDuration = this.options.duration;
                    this.options.duration = newDuration;
                    
                    // Recalculate end time
                    if (this.options.timeType === 'countdown') {
                        this.endTime = Date.now() + (newDuration * 1000);
                        this.calculateTime();
                    }
                    
                    // Show visual indicator for demo
                    this.showBehavioralTargetingIndicator(rule, behavior, oldDuration, newDuration);
                }
            }
            
            // Update colors if specified
            if (rule.colors) {
                if (rule.colors.primary) {
                    this.options.colors.primary = rule.colors.primary;
                }
                if (rule.colors.secondary) {
                    this.options.colors.secondary = rule.colors.secondary;
                }
                if (rule.colors.text) {
                    this.options.colors.text = rule.colors.text;
                }
            }
            
            // Trigger jQuery custom event
            this.$element.trigger('behavioralRuleApplied', [rule, behavior]);
        }
        
        /**
         * Show visual indicator for behavioral targeting (demo/testing only)
         * 
         * @param {Object} rule - Applied rule
         * @param {Object} behavior - Detected behavior
         * @param {number} oldDuration - Previous duration in seconds
         * @param {number} newDuration - New duration in seconds
         * @description
         * Displays a temporary badge showing which behavioral rule was applied.
         * Only visible in demo mode for testing purposes. Auto-hides after 5 seconds.
         */
        showBehavioralTargetingIndicator(rule, behavior, oldDuration, newDuration) {
            // Remove existing indicator if any
            const existingIndicator = this.$element.find('.behavioral-targeting-indicator');
            if (existingIndicator.length) {
                existingIndicator.remove();
            }
            
            // Create indicator
            const indicator = $('<div class="behavioral-targeting-indicator"></div>');
            indicator.css({
                position: 'absolute',
                top: '10px',
                left: '10px',
                background: 'rgba(102, 126, 234, 0.95)',
                color: '#fff',
                padding: '8px 12px',
                borderRadius: '6px',
                fontSize: '11px',
                fontWeight: '600',
                zIndex: '1001',
                boxShadow: '0 2px 8px rgba(0, 0, 0, 0.2)',
                fontFamily: '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif',
                maxWidth: '250px',
                lineHeight: '1.4'
            });
            
            // Build message in English
            let message = '🎯 Targeting Active: ';
            if (rule.condition === 'newVisitor') {
                message += 'New Visitor';
            } else if (rule.condition === 'returningVisitor') {
                message += `Returning Visitor (${behavior.visitCount} visits)`;
            } else if (rule.condition === behavior.device) {
                message += `Device: ${behavior.device}`;
            } else if (rule.condition === behavior.source) {
                message += `Source: ${behavior.source}`;
            } else {
                message += rule.condition;
            }
            
            message += `<br><small style="opacity: 0.9;">Duration: ${Math.round(oldDuration / 60)}min → ${Math.round(newDuration / 60)}min</small>`;
            
            indicator.html(message);
            this.$element.append(indicator);
            
            // Auto-hide after 5 seconds
            setTimeout(() => {
                indicator.fadeOut(300, function() {
                    $(this).remove();
                });
            }, 5000);
        }
        
        /**
         * Get currently applied behavioral rule
         * 
         * @returns {Object|null} Applied rule object or null if no rule applied
         * @description
         * Returns the behavioral targeting rule that is currently active.
         * Useful for debugging or displaying which rule was applied to the user.
         */
        getBehavioralRule() {
            return this.behavioralTargeting.appliedRule;
        }
        
        /**
         * Reset behavioral tracking data
         * 
         * @description
         * Clears all behavioral targeting data from localStorage and resets
         * internal state. Useful for testing or allowing users to start fresh.
         * After reset, the next visit will be treated as a new visitor.
         */
        resetBehavioralTracking() {
            try {
                const storageKey = this.behavioralTargeting.storageKey;
                localStorage.removeItem(storageKey);
                this.behavioralTargeting.visitCount = 0;
                this.behavioralTargeting.lastVisit = null;
                this.behavioralTargeting.appliedRule = null;
            } catch (e) {
                console.error('[CountdownTimerPro] Error resetting behavioral tracking:', e);
            }
        }
        
        /**
         * ============================================
         * CONVERSION OPTIMIZATION (FEATURE 3)
         * ============================================
         * Local A/B testing for automatic optimization
         */
        
        /**
         * Initialize conversion optimization system
         * FEATURE 3: Local A/B testing and automatic variant optimization
         * 
         * @description
         * Sets up the conversion optimization system that tracks user interactions and
         * conversions, then automatically optimizes timer settings (colors, duration)
         * to maximize conversion rates. Uses local A/B testing to find the best performing variant.
         */
        initAIOptimization() {
            const ai = this.options.aiOptimization;
            if (!ai || !ai.enabled) return;
            
            // Copy configuration
            this.aiOptimization.enabled = true;
            this.aiOptimization.trackInteractions = ai.trackInteractions !== false;
            this.aiOptimization.trackConversions = ai.trackConversions !== false;
            this.aiOptimization.learningRate = ai.learningRate || 0.1;
            this.aiOptimization.minSamples = ai.minSamples || 10;
            this.aiOptimization.optimizationInterval = ai.optimizationInterval || 3600000;
            this.aiOptimization.storageKey = ai.storageKey || 'countdown_timer_ai_data';
            
            // Load existing data
            this.loadAIData();
            
            // Initialize current variant
            this.initializeVariant();
            
            // Start tracking if enabled
            if (this.aiOptimization.trackInteractions) {
                this.startInteractionTracking();
            }
            
            // Start optimization interval
            this.startOptimizationInterval();
            
        }
        
        /**
         * Load AI optimization data from localStorage
         * 
         * @description
         * Restores previously saved AI optimization data including interactions,
         * conversions, variants, and statistics. Called on initialization to
         * maintain optimization state across page loads.
         */
        loadAIData() {
            try {
                const data = JSON.parse(localStorage.getItem(this.aiOptimization.storageKey) || '{}');
                this.aiOptimization.interactions = data.interactions || [];
                this.aiOptimization.conversions = data.conversions || [];
                this.aiOptimization.variants = data.variants || {};
                this.aiOptimization.lastOptimization = data.lastOptimization || null;
                this.aiOptimization.stats = data.stats || {
                    totalInteractions: 0,
                    totalConversions: 0,
                    conversionRate: 0,
                    bestVariant: null
                };
            } catch (e) {
                console.error('[CountdownTimerPro] Error loading AI data:', e);
            }
        }
        
        /**
         * Save AI optimization data to localStorage
         * 
         * @description
         * Persists AI optimization data to localStorage. Keeps only the last
         * 100 interactions and conversions to prevent localStorage bloat.
         * Called periodically and after significant events (conversions).
         */
        saveAIData() {
            try {
                const data = {
                    interactions: this.aiOptimization.interactions.slice(-100), // Keep last 100
                    conversions: this.aiOptimization.conversions.slice(-100), // Keep last 100
                    variants: this.aiOptimization.variants,
                    lastOptimization: this.aiOptimization.lastOptimization,
                    stats: this.aiOptimization.stats
                };
                localStorage.setItem(this.aiOptimization.storageKey, JSON.stringify(data));
            } catch (e) {
                console.error('[CountdownTimerPro] Error saving AI data:', e);
            }
        }
        
        /**
         * Initialize current variant for A/B testing
         * 
         * @description
         * Creates or retrieves a variant ID based on current timer configuration
         * (colors and duration). Each unique configuration gets its own variant
         * for tracking performance. Variants are stored in localStorage for
         * persistence across page loads.
         */
        initializeVariant() {
            const variantId = this.generateVariantId();
            
            // Check if variant exists
            if (!this.aiOptimization.variants[variantId]) {
                this.aiOptimization.variants[variantId] = {
                    id: variantId,
                    colors: {
                        primary: this.options.colors.primary,
                        secondary: this.options.colors.secondary,
                        text: this.options.colors.text
                    },
                    duration: this.options.duration,
                    interactions: 0,
                    conversions: 0,
                    conversionRate: 0,
                    createdAt: Date.now()
                };
            }
            
            this.aiOptimization.currentVariant = variantId;
        }
        
        /**
         * Generate variant ID based on current configuration
         * 
         * @returns {string} Unique variant ID
         * @description
         * Creates a unique identifier for the current timer configuration by
         * hashing the colors and duration. This ensures that identical configurations
         * share the same variant ID for accurate performance tracking.
         */
        generateVariantId() {
            const colors = this.options.colors;
            const duration = this.options.duration;
            // Create a simple hash of the configuration
            const config = `${colors.primary}-${colors.secondary}-${duration}`;
            return 'variant_' + this.simpleHash(config);
        }
        
        /**
         * Simple hash function for generating variant IDs
         * 
         * @param {string} str - String to hash
         * @returns {string} Hashed string
         * @description
         * Generates a simple hash from a string. Used to create unique variant IDs
         * from timer configurations. Returns a base36 encoded hash for readability.
         */
        simpleHash(str) {
            let hash = 0;
            for (let i = 0; i < str.length; i++) {
                const char = str.charCodeAt(i);
                hash = ((hash << 5) - hash) + char;
                hash = hash & hash; // Convert to 32bit integer
            }
            return Math.abs(hash).toString(36);
        }
        
        /**
         * Start tracking user interactions
         * 
         * @description
         * Sets up event listeners to automatically track user interactions with
         * the timer. Tracks clicks and views (when timer becomes visible) for
         * AI optimization analysis. Uses IntersectionObserver for view tracking.
         */
        startInteractionTracking() {
            // Track clicks on the timer
            this.$element.on('click', () => {
                this.trackInteraction('click');
            });
            
            // Track when timer is visible (impression)
            if (typeof IntersectionObserver !== 'undefined') {
                const observer = new IntersectionObserver((entries) => {
                    entries.forEach(entry => {
                        if (entry.isIntersecting) {
                            this.trackInteraction('view');
                        }
                    });
                }, { threshold: 0.5 });
                
                observer.observe(this.element);
            }
        }
        
        /**
         * Track user interaction for AI optimization
         * 
         * @param {string} type - Type of interaction ('click', 'view', etc.)
         * @description
         * Records user interactions (clicks, views) to analyze which timer
         * configurations perform best. Data is stored in localStorage and used
         * for automatic optimization.
         */
        trackInteraction(type) {
            if (!this.aiOptimization.enabled || !this.aiOptimization.trackInteractions) return;
            
            const interaction = {
                type: type,
                variantId: this.aiOptimization.currentVariant,
                timestamp: Date.now(),
                remainingTime: this.remainingTime.total
            };
            
            this.aiOptimization.interactions.push(interaction);
            this.aiOptimization.stats.totalInteractions++;
            
            // Update variant stats
            if (this.aiOptimization.variants[this.aiOptimization.currentVariant]) {
                this.aiOptimization.variants[this.aiOptimization.currentVariant].interactions++;
            }
            
            // Save data periodically (every 10 interactions)
            if (this.aiOptimization.interactions.length % 10 === 0) {
                this.saveAIData();
            }
        }
        
        /**
         * Track conversion for AI optimization
         * 
         * @param {string} source - Source of conversion ('manual', 'cta-click', etc.)
         * @description
         * Records a conversion event (e.g., user clicks CTA, completes purchase).
         * This is used to calculate conversion rates for each timer variant and
         * automatically optimize to the best performing configuration.
         * 
         * @example
         * // Track conversion when user clicks CTA button
         * $('#my-timer').data('countdownTimerPro').trackConversion('cta-click');
         */
        trackConversion(source = 'manual') {
            if (!this.aiOptimization.enabled || !this.aiOptimization.trackConversions) return;
            
            const conversion = {
                variantId: this.aiOptimization.currentVariant,
                timestamp: Date.now(),
                source: source,
                remainingTime: this.remainingTime.total
            };
            
            this.aiOptimization.conversions.push(conversion);
            this.aiOptimization.stats.totalConversions++;
            
            // Update variant stats
            const variant = this.aiOptimization.variants[this.aiOptimization.currentVariant];
            if (variant) {
                variant.conversions++;
                variant.conversionRate = variant.conversions / Math.max(1, variant.interactions);
            }
            
            // Update overall stats
            this.aiOptimization.stats.conversionRate = 
                this.aiOptimization.stats.totalConversions / Math.max(1, this.aiOptimization.stats.totalInteractions);
            
            // Save data
            this.saveAIData();
            
            // Trigger event
            this.$element.trigger('aiConversion', [conversion]);
        }
        
        /**
         * Start optimization interval
         * 
         * @description
         * Sets up a periodic interval to run optimization checks. The interval
         * is configurable via optimizationInterval option (default: 1 hour).
         * Optimization only runs if minimum sample size is reached.
         */
        startOptimizationInterval() {
            if (this.aiOptimization.optimizationIntervalId) {
                clearInterval(this.aiOptimization.optimizationIntervalId);
            }
            
            this.aiOptimization.optimizationIntervalId = setInterval(() => {
                this.optimizeNow();
            }, this.aiOptimization.optimizationInterval);
        }
        
        /**
         * Run optimization immediately
         * 
         * @description
         * Analyzes collected data and applies the best performing variant.
         * Only runs if minimum sample size (minSamples) is reached.
         * Can be called manually or runs automatically at optimizationInterval.
         */
        optimizeNow() {
            if (!this.aiOptimization.enabled) return;
            
            // Check if we have enough samples
            if (this.aiOptimization.stats.totalInteractions < this.aiOptimization.minSamples) {
                return;
            }
            
            // Find best variant
            const bestVariant = this.findBestVariant();
            
            if (bestVariant && bestVariant.id !== this.aiOptimization.currentVariant) {
                // Apply best variant
                this.applyVariant(bestVariant);
            }
            
            this.aiOptimization.lastOptimization = Date.now();
            this.saveAIData();
        }
        
        /**
         * Find best variant based on conversion rate
         * 
         * @returns {Object|null} Best performing variant or null if none found
         * @description
         * Compares all variants and returns the one with the highest conversion rate.
         * Only considers variants with enough samples (minSamples) to ensure
         * statistical significance.
         */
        findBestVariant() {
            let bestVariant = null;
            let bestRate = 0;
            
            for (const variantId in this.aiOptimization.variants) {
                const variant = this.aiOptimization.variants[variantId];
                // Only consider variants with enough samples
                if (variant.interactions >= this.aiOptimization.minSamples) {
                    if (variant.conversionRate > bestRate) {
                        bestRate = variant.conversionRate;
                        bestVariant = variant;
                    }
                }
            }
            
            this.aiOptimization.stats.bestVariant = bestVariant ? bestVariant.id : null;
            return bestVariant;
        }
        
        /**
         * Apply variant to timer
         * 
         * @param {Object} variant - Variant object with colors and duration
         * @description
         * Applies the optimized variant by updating timer colors and duration.
         * Triggers 'aiOptimized' event when variant is applied.
         */
        applyVariant(variant) {
            if (!variant) return;
            
            // Update colors
            if (variant.colors) {
                this.options.colors.primary = variant.colors.primary;
                this.options.colors.secondary = variant.colors.secondary;
                if (variant.colors.text) {
                    this.options.colors.text = variant.colors.text;
                }
            }
            
            // Update duration (if countdown mode)
            if (variant.duration && this.options.timeType === 'countdown') {
                const oldDuration = this.options.duration;
                this.options.duration = variant.duration;
                
                // Recalculate end time
                if (this.endTime) {
                    const timeElapsed = Date.now() - (this.endTime - (oldDuration * 1000));
                    this.endTime = Date.now() + (variant.duration * 1000) - timeElapsed;
                    this.calculateTime();
                }
            }
            
            this.aiOptimization.currentVariant = variant.id;
            
            // Trigger event
            this.$element.trigger('aiOptimized', [variant]);
        }
        
        /**
         * Get AI optimization statistics
         * 
         * @returns {Object} Statistics object with conversion rates and variant data
         * @description
         * Returns current AI optimization statistics including total interactions,
         * conversions, conversion rates, and best performing variant.
         * 
         * @example
         * const stats = $('#my-timer').data('countdownTimerPro').getAIStats();
         * console.log('Conversion Rate:', stats.stats.conversionRate);
         */
        getAIStats() {
            return {
                enabled: this.aiOptimization.enabled,
                currentVariant: this.aiOptimization.currentVariant,
                stats: this.aiOptimization.stats,
                variants: this.aiOptimization.variants,
                lastOptimization: this.aiOptimization.lastOptimization
            };
        }
        
        /**
         * Reset AI optimization data
         * 
         * @description
         * Clears all AI optimization data from localStorage and resets statistics.
         * Useful for testing or starting fresh optimization cycle.
         */
        resetAIData() {
            try {
                localStorage.removeItem(this.aiOptimization.storageKey);
                this.aiOptimization.interactions = [];
                this.aiOptimization.conversions = [];
                this.aiOptimization.variants = {};
                this.aiOptimization.currentVariant = null;
                this.aiOptimization.lastOptimization = null;
                this.aiOptimization.stats = {
                    totalInteractions: 0,
                    totalConversions: 0,
                    conversionRate: 0,
                    bestVariant: null
                };
                this.initializeVariant();
            } catch (e) {
                console.error('[CountdownTimerPro] Error resetting AI data:', e);
            }
        }
        
        /**
         * Create background pattern canvas
         */
        createBackgroundPattern() {
            if (this.patternCanvas) return;
            
            this.patternCanvas = document.createElement('canvas');
            this.patternCtx = this.patternCanvas.getContext('2d');
            this.updateBackgroundPattern();
        }
        
        /**
         * Update background pattern
         */
        updateBackgroundPattern() {
            if (!this.patternCanvas || !this.patternCtx) return;
            
            const pattern = this.options.backgroundPattern;
            if (!pattern.enabled || pattern.type === 'none') return;
            
            const size = pattern.size || 20;
            const spacing = pattern.spacing || 10;
            const cellSize = size + spacing;
            const canvasSize = cellSize * 4; // Create a repeating pattern
            
            this.patternCanvas.width = canvasSize;
            this.patternCanvas.height = canvasSize;
            
            this.patternCtx.clearRect(0, 0, canvasSize, canvasSize);
            this.patternCtx.strokeStyle = pattern.color;
            this.patternCtx.fillStyle = pattern.color;
            this.patternCtx.lineWidth = 1;
            
            switch (pattern.type) {
                case 'grid':
                    for (let x = 0; x <= canvasSize; x += cellSize) {
                        this.patternCtx.beginPath();
                        this.patternCtx.moveTo(x, 0);
                        this.patternCtx.lineTo(x, canvasSize);
                        this.patternCtx.stroke();
                    }
                    for (let y = 0; y <= canvasSize; y += cellSize) {
                        this.patternCtx.beginPath();
                        this.patternCtx.moveTo(0, y);
                        this.patternCtx.lineTo(canvasSize, y);
                        this.patternCtx.stroke();
                    }
                    break;
                    
                case 'dots':
                    for (let x = cellSize / 2; x < canvasSize; x += cellSize) {
                        for (let y = cellSize / 2; y < canvasSize; y += cellSize) {
                            this.patternCtx.beginPath();
                            this.patternCtx.arc(x, y, 1.5, 0, Math.PI * 2);
                            this.patternCtx.fill();
                        }
                    }
                    break;
                    
                case 'lines':
                    for (let x = 0; x < canvasSize; x += cellSize) {
                        this.patternCtx.beginPath();
                        this.patternCtx.moveTo(x, 0);
                        this.patternCtx.lineTo(x, canvasSize);
                        this.patternCtx.stroke();
                    }
                    break;
                    
                case 'diagonal':
                    for (let i = -canvasSize; i < canvasSize * 2; i += cellSize) {
                        this.patternCtx.beginPath();
                        this.patternCtx.moveTo(i, 0);
                        this.patternCtx.lineTo(i + canvasSize, canvasSize);
                        this.patternCtx.stroke();
                    }
                    break;
            }
        }

        /**
         * Draw the timer on canvas
         */
        draw() {
            const ctx = this.ctx;
            const dpr = window.devicePixelRatio || 1;
            const width = this.canvas.width / dpr;
            const height = this.canvas.height / dpr;
            
            // Clear canvas
            ctx.clearRect(0, 0, width, height);
            
            // Draw background pattern if enabled (draw first, then draw timer on top)
            if (this.options.backgroundPattern && this.options.backgroundPattern.enabled && this.patternCanvas) {
                ctx.save();
                const pattern = ctx.createPattern(this.patternCanvas, 'repeat');
                if (pattern) {
                    ctx.fillStyle = pattern;
                    ctx.fillRect(0, 0, width, height);
                }
                ctx.restore();
            }
            
            // Get current colors
            const colors = this.getCurrentColors();
            
            // Prepare time units
            const timeUnits = [
                { value: this.remainingTime.days, label: this.options.texts.days, show: this.options.showDays, max: 365 },
                { value: this.remainingTime.hours, label: this.options.texts.hours, show: this.options.showHours, max: 24 },
                { value: this.remainingTime.minutes, label: this.options.texts.minutes, show: this.options.showMinutes, max: 60 },
                { value: this.remainingTime.seconds, label: this.options.texts.seconds, show: this.options.showSeconds, max: 60 }
            ];
            
            const visibleUnits = timeUnits.filter(unit => unit.show);
            const totalVisible = visibleUnits.length;
            
            if (totalVisible === 0) return;
            
            // Determine layout: always horizontal for better visibility
            // Mobile uses vertical layout when width < 600px
            // CRITICAL: Use this.options.width for mobile detection, not canvas width
            const originalWidth = this.options.width || width;
            const checkWidth = this.options.width || width;
            const isMobile = checkWidth < 600;
            // Use vertical layout for mobile (width < 600px) to show all circles properly
            const useVerticalLayout = isMobile;
            
            // Adaptive padding based on screen size (use checkWidth for mobile detection)
            let padding;
            if (checkWidth < 600) {
                // Mobile: minimal padding but enough for visibility
                padding = Math.max(8, Math.min(15, checkWidth * 0.015)); // 1.5% padding, min 8px, max 15px
            } else if (width >= 1920) {
                // Large screens - more padding for better spacing
                padding = Math.max(40, Math.min(80, width * 0.06));
            } else if (width >= 1024) {
                // Desktop - moderate padding
                padding = Math.max(30, Math.min(60, width * 0.05));
            } else {
                padding = Math.max(20, Math.min(40, width * 0.05));
            }
            const availableWidth = Math.max(200, width - (padding * 2));
            const availableHeight = Math.max(200, height - (padding * 2));
            
            // Calculate circle size with better responsive handling (use original width for mobile detection)
            // Mobile: much smaller circles for better proportions (30px minimum diameter)
            const minCircleSize = checkWidth < 600 ? 30 : (width >= 1920 ? 120 : (width >= 1024 ? 100 : 60));
            let maxCircleSize, circleSize, circleRadius;
            
            if (useVerticalLayout) {
                // Vertical layout: circles stacked vertically (mobile)
                // Optimized for mobile: larger circles, better spacing, properly centered
                // Use 85% of available width for larger circles (was 75%)
                const maxWidthBased = Math.min(availableWidth * 0.85, 150); // Max 85% of width, cap at 150px radius
                
                // Calculate optimal circle size for mobile - larger and more visible
                // Use viewport height to ensure all circles fit comfortably
                // On mobile, allow full vertical expansion to show all circles
                const viewportHeight = window.innerHeight || availableHeight;
                const maxHeightForCircles = Math.max(viewportHeight * 0.85, availableHeight * 0.9); // Use 85% of viewport or 90% of available
                
                // Calculate spacing - better spacing on mobile (12% instead of 10%)
                const estimatedCircleSize = Math.min(
                    maxWidthBased, 
                    maxHeightForCircles / (totalVisible * 1.3) // Less compact: 1.3 instead of 1.4 for larger circles
                );
                
                // Better spacing for mobile
                const spacingBetween = Math.max(12, estimatedCircleSize * 0.12); // 12% spacing, minimum 12px
                
                // Calculate total space needed
                const totalSpacing = spacingBetween * (totalVisible - 1);
                const availableForCircles = maxHeightForCircles - totalSpacing;
                const circleHeight = availableForCircles / totalVisible;
                
                // Final circle size - larger for better visibility
                maxCircleSize = Math.min(maxWidthBased, circleHeight * 0.95);
                // Increased size range: 60px to 140px diameter (was 50-100px)
                circleSize = Math.max(60, Math.min(maxCircleSize, 140)); // Between 60px and 140px diameter
                circleRadius = Math.max(30, Math.min(circleSize / 2, 70)); // Between 30px and 70px radius
                
                // Recalculate spacing with actual circle size - better spacing
                const actualSpacing = Math.max(12, Math.min(25, (maxHeightForCircles - (circleRadius * 2 * totalVisible)) / (totalVisible + 1)));
                
                // Calculate total height needed - use better padding on mobile for centering
                // Mobile: more top padding to push circles down, less bottom padding
                const mobilePaddingTop = Math.max(60, Math.min(80, width * 0.12)); // 12% top padding, min 60px, max 80px - pushes circles down
                const mobilePaddingBottom = Math.max(15, Math.min(30, width * 0.03)); // 3% bottom padding, min 15px, max 30px
                const totalHeight = (circleRadius * 2 * totalVisible) + (actualSpacing * (totalVisible - 1)) + mobilePaddingTop + mobilePaddingBottom;
                
                // CRITICAL: On mobile, ALWAYS ensure canvas height adapts to content to prevent clipping
                // Always update canvas height for vertical layout to ensure all circles are visible
                const dpr = window.devicePixelRatio || 1;
                const requiredHeight = Math.ceil(totalHeight);
                
                // ALWAYS update canvas height for mobile vertical layout - force expansion
                // This ensures the container expands vertically to show all circles
                const heightChanged = Math.abs(height - requiredHeight) > 1 || height < requiredHeight;
                
                if (heightChanged) {
                    // Update canvas height - force expansion
                    this.canvas.height = requiredHeight * dpr;
                    this.canvas.style.height = requiredHeight + 'px';
                    // Update context scale
                    this.ctx.setTransform(1, 0, 0, 1, 0, 0);
                    this.ctx.scale(dpr, dpr);
                    // Clear and redraw with new height
                    ctx.clearRect(0, 0, width, requiredHeight);
                }
                
                // Always draw circles (whether height changed or not)
                // Center vertically: start from top padding + radius (more space above for "People Watching")
                const startY = mobilePaddingTop + circleRadius;
                // Center horizontally: use exact center of canvas width - PERFECT CENTERING
                // CRITICAL: Use the actual rendered canvas width (not style width) for perfect centering
                // The canvas width variable already accounts for devicePixelRatio
                // Use width directly from draw() method which is already calculated correctly
                const centerX = width / 2;
                
                // Draw each circle vertically
                visibleUnits.forEach((unit, index) => {
                    const circleY = startY + (index * ((circleRadius * 2) + actualSpacing));
                    this.drawCircle(ctx, unit, index, centerX, circleY, circleRadius, colors, totalVisible);
                });
                
                // CRITICAL: Force resize after canvas height update to ensure container expands
                // This must happen after drawing to ensure the canvas has the correct size
                if (heightChanged) {
                    // Use requestAnimationFrame to ensure DOM has updated
                    requestAnimationFrame(() => {
                        // Force resize to update container
                        this.resize();
                        // Also trigger a redraw to ensure everything is correct
                        setTimeout(() => {
                            this.draw();
                        }, 50);
                    });
                }
            } else {
                // Horizontal layout: circles side by side (desktop)
                maxCircleSize = (height - padding * 2) * 0.55;
                // Better spacing for large screens
                let spacingBetween;
                if (width >= 1920) {
                    spacingBetween = Math.max(40, Math.min(80, width * 0.04));
                } else if (width >= 1024) {
                    spacingBetween = Math.max(30, Math.min(60, width * 0.035));
                } else if (checkWidth < 600) {
                    spacingBetween = 10;
                } else {
                    spacingBetween = Math.max(15, width * 0.03);
                }
                const widthBasedSize = (availableWidth / totalVisible) - spacingBetween;
                circleSize = Math.max(minCircleSize, Math.min(widthBasedSize, maxCircleSize));
                circleRadius = Math.max(25, circleSize / 2);
                
                const spacing = availableWidth / totalVisible;
                const startX = padding + (spacing / 2);
                const centerY = height / 2;
                
                // Draw each circle horizontally
                visibleUnits.forEach((unit, index) => {
                    const circleX = startX + (index * spacing);
                    const circleY = centerY;
                    this.drawCircle(ctx, unit, index, circleX, circleY, circleRadius, colors, totalVisible);
                });
            }
            
            // Update animation progress for color transitions
            if (this.options.multicolor.enabled) {
                this.animationProgress += 0.005;
                if (this.animationProgress >= 1) {
                    this.animationProgress = 0;
                    this.colorIndex = (this.colorIndex + 1) % this.options.multicolor.colors.length;
                }
            }
            
            // Update pulse/glow animation progress
            // CRITICAL: Ensure pulseGlow exists and is properly initialized
            if (!this.options.pulseGlow) {
                this.options.pulseGlow = {
                    enabled: false,
                    intensity: 1.0,
                    speed: 1.0
                };
            }
            if (this.options.pulseGlow && this.options.pulseGlow.enabled) {
                this.pulseProgress += 0.02 * (this.options.pulseGlow.speed || 1.0);
                if (this.pulseProgress >= Math.PI * 2) {
                    this.pulseProgress = 0;
                }
            }
            
            // Update gradient wave animation progress
            if (this.options.gradientWave.enabled) {
                this.gradientWaveProgress += 0.01 * this.options.gradientWave.speed;
                if (this.gradientWaveProgress >= 1) {
                    this.gradientWaveProgress = 0;
                }
            }
            
            // Update liquid effect wave animation - isolated from other animations
            // This animation is independent and must not be affected by multicolor, pulseGlow, or gradientWave
            const liquidEffectEnabled = (this.options.liquidEffect && this.options.liquidEffect.enabled !== false) || 
                                       (!this.options.liquidEffect); // Default to true if not specified
            
            if (liquidEffectEnabled) {
                // Ensure liquidEffect exists
                if (!this.options.liquidEffect) {
                    this.options.liquidEffect = {
                        enabled: true,
                        intensity: 1.2,
                        bubbleSpeed: 1.0,
                        waveSpeed: 1.0,
                        bubbleSpawnRate: 0.8,
                        showEvaporation: true,
                        liquidColor: null
                    };
                }
                
                const waveSpeed = this.options.liquidEffect.waveSpeed || 1.0;
                
                // CRITICAL: Initialize and preserve liquidWaveOffset - COMPLETELY ISOLATED from other animations
                // This ensures waves continue smoothly even when preset changes
                // Never reset this value - it's independent of multicolor, pulseGlow, gradientWave animations
                if (this.liquidWaveOffset === undefined || 
                    this.liquidWaveOffset === null || 
                    isNaN(this.liquidWaveOffset) ||
                    !isFinite(this.liquidWaveOffset)) {
                    // Check if we have a preserved value from options (preset change)
                    if (this.options._preservedLiquidWaveOffset !== undefined && 
                        !isNaN(this.options._preservedLiquidWaveOffset) && 
                        isFinite(this.options._preservedLiquidWaveOffset)) {
                        this.liquidWaveOffset = this.options._preservedLiquidWaveOffset;
                    } else {
                        // Only initialize if truly undefined/null/NaN, preserve existing value otherwise
                        // Never reset to 0 if it already has a value
                        this.liquidWaveOffset = this.liquidWaveOffset || 0;
                    }
                }
                
                // Always increment from existing value - never reset
                // This animation is INDEPENDENT and must continue smoothly regardless of other animations
                // Use a small increment for smooth, realistic wave animation (0.015 = slow and realistic)
                const currentOffset = (this.liquidWaveOffset !== undefined && 
                                      this.liquidWaveOffset !== null && 
                                      !isNaN(this.liquidWaveOffset) &&
                                      isFinite(this.liquidWaveOffset)) 
                                     ? this.liquidWaveOffset : 0;
                this.liquidWaveOffset = currentOffset + (0.015 * waveSpeed);
                
                // Use modulo to keep it within reasonable range but preserve continuity
                // Only wrap when it gets very large to prevent overflow
                if (this.liquidWaveOffset > Math.PI * 20) {
                    // Preserve the fractional part to maintain smooth animation
                    const wrapped = this.liquidWaveOffset % (Math.PI * 2);
                    this.liquidWaveOffset = wrapped;
                }
            } else {
                // Even if disabled, keep the offset value to prevent reset when re-enabled
                // Never reset liquidWaveOffset - preserve it even when disabled
                if (this.liquidWaveOffset === undefined || 
                    this.liquidWaveOffset === null || 
                    isNaN(this.liquidWaveOffset) ||
                    !isFinite(this.liquidWaveOffset)) {
                    // Only initialize if truly missing, never reset existing value
                    this.liquidWaveOffset = this.liquidWaveOffset || 0;
                }
            }
        }
        
        /**
         * Draw a single circle with all effects
         */
        drawCircle(ctx, unit, index, circleX, circleY, circleRadius, colors, totalVisible) {
            const currentColor = colors[index % colors.length];
            
            // Calculate progress with smooth animation
            let progress = 0;
            if (this.options.timeType === 'countdown') {
                progress = Math.min(1, unit.value / unit.max);
            } else {
                // For countup, calculate progress based on total duration
                if (this.options.duration && this.options.duration > 0) {
                    // Calculate what portion of total duration this unit represents
                    const maxDuration = this.options.duration * 1000; // Convert to milliseconds
                    const totalElapsed = this.remainingTime.total;
                    
                    // Calculate progress for this specific unit based on total duration
                    // Each unit should show progress relative to its portion of total duration
                    if (index === totalVisible - 1) {
                        // Last unit (usually seconds) - show progress based on total duration
                        progress = Math.min(1, totalElapsed / maxDuration);
                    } else {
                        // For other units, calculate based on their portion
                        // Convert total elapsed to the unit's scale
                        let unitTotal = 0;
                        if (unit.label === this.options.texts.days) {
                            unitTotal = Math.floor(totalElapsed / (1000 * 60 * 60 * 24));
                        } else if (unit.label === this.options.texts.hours) {
                            unitTotal = Math.floor((totalElapsed % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
                        } else if (unit.label === this.options.texts.minutes) {
                            unitTotal = Math.floor((totalElapsed % (1000 * 60 * 60)) / (1000 * 60));
                        }
                        progress = Math.min(1, unitTotal / unit.max);
                    }
                } else {
                    // No duration limit - show progress within the max value (cyclic)
                    progress = (unit.value % unit.max) / unit.max;
                }
            }
            
            // Add smooth easing
            const easedProgress = this.easeInOutQuad(progress);
            
            // Apply 3D effects (parallax offset)
            let parallaxOffsetX = 0;
            let parallaxOffsetY = 0;
            let shadowDepth = 2;
            let shadowBlurBase = 10;
            
            if (this.options.effects3D.enabled) {
                const parallaxIntensity = this.options.effects3D.parallaxIntensity || 0;
                const depth = this.options.effects3D.shadowDepth || 20;
                
                // Parallax effect based on position
                parallaxOffsetX = (circleX - (this.canvas.width / (window.devicePixelRatio || 1)) / 2) * parallaxIntensity * 0.1;
                parallaxOffsetY = (circleY - (this.canvas.height / (window.devicePixelRatio || 1)) / 2) * parallaxIntensity * 0.1;
                
                // Shadow depth
                shadowDepth = depth * 0.1;
                shadowBlurBase = 10 + (depth * 0.3);
            }
            
            // Draw circle background with 3D shadow
            ctx.save();
            if (this.options.effects3D.enabled) {
                ctx.shadowColor = 'rgba(0, 0, 0, 0.3)';
                ctx.shadowBlur = shadowBlurBase;
                ctx.shadowOffsetX = shadowDepth;
                ctx.shadowOffsetY = shadowDepth;
            } else {
                ctx.shadowColor = 'rgba(0, 0, 0, 0.1)';
                ctx.shadowBlur = 10;
                ctx.shadowOffsetX = 0;
                ctx.shadowOffsetY = 2;
            }
            
            ctx.beginPath();
            ctx.arc(circleX + parallaxOffsetX, circleY + parallaxOffsetY, circleRadius, 0, Math.PI * 2);
            const bgColor = this.options.colors.circleBackground;
            ctx.strokeStyle = bgColor;
            // Reduce border thickness on mobile for better proportions
            const isMobileCircle = circleRadius < 35; // Mobile circles are smaller
            const borderThickness = isMobileCircle 
                ? Math.max(2, Math.min(4, this.options.circleThickness * 0.3)) // 30% of original, min 2px, max 4px
                : this.options.circleThickness;
            ctx.lineWidth = borderThickness;
            ctx.lineJoin = 'round';
            ctx.lineCap = 'round';
            ctx.stroke();
            ctx.restore();
            
            // Draw liquid effect (always enabled - distinctive visual feature)
            if (!this.options.liquidEffect) {
                this.options.liquidEffect = {
                    enabled: true,
                    intensity: 1.2,
                    bubbleSpeed: 1.0,
                    waveSpeed: 1.0,
                    bubbleSpawnRate: 0.8,
                    showEvaporation: true,
                    liquidColor: null
                };
            }
            this.options.liquidEffect.enabled = true;
            
            this.drawLiquidEffect(ctx, unit, index, circleX, circleY, circleRadius, currentColor, progress, easedProgress, parallaxOffsetX, parallaxOffsetY);
            
            // Draw inner glow effect
            ctx.save();
            ctx.beginPath();
            // Use reduced border thickness for mobile
            const isMobileGlow = circleRadius < 35;
            const borderThicknessForGlow = isMobileGlow 
                ? Math.max(2, Math.min(4, this.options.circleThickness * 0.3))
                : this.options.circleThickness;
            ctx.arc(circleX, circleY, circleRadius - (borderThicknessForGlow / 2), 0, Math.PI * 2);
            const innerGradient = ctx.createRadialGradient(
                circleX, circleY, 0,
                circleX, circleY, circleRadius
            );
            innerGradient.addColorStop(0, 'rgba(255, 255, 255, 0.3)');
            innerGradient.addColorStop(1, 'rgba(255, 255, 255, 0)');
            ctx.fillStyle = innerGradient;
            ctx.fill();
            ctx.restore();
            
            // Draw value text with shadow (responsive font size - always fits in circle)
            ctx.save();
            ctx.shadowColor = 'rgba(0, 0, 0, 0.3)';
            ctx.shadowBlur = 5;
            ctx.shadowOffsetX = 0;
            ctx.shadowOffsetY = 2;
            
            ctx.fillStyle = this.options.colors.text;
            
            // Determine if vertical layout (mobile) - check if circles are stacked
            // Check both actual canvas width and original width option to handle responsive scaling
            const dpr = window.devicePixelRatio || 1;
            const canvasWidth = this.canvas.width / dpr;
            const originalWidth = this.options.width || canvasWidth;
            const isVerticalLayout = canvasWidth < 600 || originalWidth < 600 || 
                                    (this.canvas.height / dpr > canvasWidth && canvasWidth < 800);
            
            // Calculate font size based on circle radius to ensure it ALWAYS fits
            // Be VERY conservative - text must fit comfortably inside the circle
            // Available vertical space: circleRadius * 2 (diameter) minus padding
            const verticalPadding = circleRadius * 0.4; // 40% padding top and bottom (very conservative)
            const availableVerticalSpace = (circleRadius * 2) - (verticalPadding * 2);
            
            // Number should take max 45% of available vertical space (conservative)
            // On mobile, use smaller font size for better fit
            const maxNumberHeight = availableVerticalSpace * 0.45;
            const preferredNumberSize = isVerticalLayout && originalWidth < 600 
                ? this.options.fontSize * 0.8  // Smaller font on mobile
                : this.options.fontSize * 1.2;
            // Use the smaller of the two, and be very conservative (80% of max)
            let numberFontSize = Math.max(12, Math.min(maxNumberHeight * 0.75, preferredNumberSize));
            
            const numberWeight = this.options.fontWeight || 'bold';
            ctx.font = `${numberWeight} ${numberFontSize}px ${this.options.fontFamily}`;
            ctx.textAlign = 'center';
            ctx.textBaseline = 'middle';
            
            // Measure text to ensure it fits both width and height
            const valueText = String(unit.value).padStart(2, '0');
            const textMetrics = ctx.measureText(valueText);
            const textWidth = textMetrics.width;
            const textHeight = numberFontSize * 1.2; // Account for line height
            
            // Check both width and height constraints - be very conservative
            const maxTextWidth = circleRadius * 1.2; // Max width is 60% of diameter (very conservative)
            const maxTextHeight = availableVerticalSpace * 0.45; // Max height is 45% of available space
            
            let finalNumberFontSize = numberFontSize;
            // Scale down if too wide
            if (textWidth > maxTextWidth) {
                finalNumberFontSize = (numberFontSize * maxTextWidth) / textWidth;
            }
            // Scale down if too tall
            if (textHeight > maxTextHeight) {
                finalNumberFontSize = Math.min(finalNumberFontSize, maxTextHeight / 1.2);
            }
            
            // Ensure minimum readable size
            finalNumberFontSize = Math.max(12, finalNumberFontSize);
            ctx.font = `${numberWeight} ${finalNumberFontSize}px ${this.options.fontFamily}`;
            
            // Position number higher - use moderate offset
            const numberOffset = Math.max(5, finalNumberFontSize * 0.25);
            ctx.fillText(valueText, circleX, circleY - numberOffset);
            ctx.restore();
            
            // Draw label text (responsive font size - always fits in circle)
            ctx.save();
            ctx.fillStyle = this.options.colors.text;
            ctx.globalAlpha = 0.75;
            
            // Label should take max 25% of available vertical space (very conservative)
            const maxLabelHeight = availableVerticalSpace * 0.25;
            const preferredLabelSize = this.options.fontSize * 0.45;
            let labelFontSize = Math.max(8, Math.min(maxLabelHeight * 0.8, preferredLabelSize));
            
            const labelWeight = this.options.labelFontWeight || 'normal';
            ctx.font = `${labelWeight} ${labelFontSize}px ${this.options.fontFamily}`;
            ctx.textAlign = 'center';
            ctx.textBaseline = 'middle';
            
            // Measure label text
            const labelMetrics = ctx.measureText(unit.label);
            const labelWidth = labelMetrics.width;
            const labelHeight = labelFontSize * 1.2;
            const maxLabelWidth = circleRadius * 1.2; // Very conservative width (60% of diameter)
            
            let finalLabelFontSize = labelFontSize;
            // Scale down if too wide
            if (labelWidth > maxLabelWidth) {
                finalLabelFontSize = (labelFontSize * maxLabelWidth) / labelWidth;
            }
            // Scale down if too tall
            if (labelHeight > maxLabelHeight) {
                finalLabelFontSize = Math.min(finalLabelFontSize, maxLabelHeight / 1.2);
            }
            
            // Ensure minimum readable size
            finalLabelFontSize = Math.max(8, finalLabelFontSize);
            ctx.font = `${labelWeight} ${finalLabelFontSize}px ${this.options.fontFamily}`;
            
            // Position label lower - moderate spacing from number
            // Calculate spacing based on both font sizes with moderate padding
            const spacingBetween = Math.max(8, finalNumberFontSize * 0.35 + finalLabelFontSize * 0.3);
            const labelOffset = numberOffset + spacingBetween;
            
            // Make sure label doesn't go outside circle
            const maxLabelPosition = circleY + circleRadius - 8;
            const finalLabelOffset = Math.min(labelOffset, maxLabelPosition - circleY);
            
            ctx.fillText(
                unit.label,
                circleX,
                circleY + finalLabelOffset
            );
            ctx.restore();
        }
        
        /**
         * Draw liquid effect with bubbles and evaporation
         * Distinctive visual feature - Liquid that consumes with bubbles as it evaporates
         * 
         * @param {CanvasRenderingContext2D} ctx - Canvas context
         * @param {Object} unit - Time unit object
         * @param {number} index - Unit index
         * @param {number} circleX - Circle center X
         * @param {number} circleY - Circle center Y
         * @param {number} circleRadius - Circle radius
         * @param {string} currentColor - Current color
         * @param {number} progress - Progress (0 to 1)
         * @param {number} easedProgress - Eased progress
         * @param {number} parallaxOffsetX - Parallax X offset
         * @param {number} parallaxOffsetY - Parallax Y offset
         */
        drawLiquidEffect(ctx, unit, index, circleX, circleY, circleRadius, currentColor, progress, easedProgress, parallaxOffsetX, parallaxOffsetY) {
            ctx.save();
            
            // Ensure liquidEffect options exist (use defaults if not specified)
            if (!this.options.liquidEffect) {
                this.options.liquidEffect = {
                    enabled: true,
                    intensity: 1.2,
                    bubbleSpeed: 1.0,
                    waveSpeed: 1.0,
                    bubbleSpawnRate: 0.8,
                    showEvaporation: true,
                    liquidColor: null
                };
            }
            
            // Ensure liquidWaveOffset is initialized (prevents waves from disappearing)
            if (this.liquidWaveOffset === undefined || this.liquidWaveOffset === null || isNaN(this.liquidWaveOffset)) {
                this.liquidWaveOffset = 0;
            }
            
            // CRITICAL: Get liquid color - STABILIZE when multicolor is active
            // When multicolor is enabled, currentColor changes dynamically, causing waves to disappear/appear
            // We need to use a STABLE color for the liquid to prevent this issue
            let liquidColor;
            if (this.options.liquidEffect.liquidColor) {
                // Use custom liquid color if specified
                liquidColor = this.options.liquidEffect.liquidColor;
            } else if (this.options.multicolor && this.options.multicolor.enabled && this.options.multicolor.colors && this.options.multicolor.colors.length > 0) {
                // CRITICAL: When multicolor is active, use the FIRST color from multicolor array
                // This prevents the liquid color from changing dynamically and causing waves to disappear
                liquidColor = this.options.multicolor.colors[0];
            } else {
                // Use currentColor as fallback (only when multicolor is not active)
                liquidColor = currentColor;
            }
            
            // Ensure liquidColor is valid (not white, not empty, not null)
            if (!liquidColor || liquidColor === '#ffffff' || liquidColor === '#fff' || liquidColor === 'white') {
                // Fallback to circle color or first multicolor color
                if (this.options.multicolor && this.options.multicolor.enabled && this.options.multicolor.colors && this.options.multicolor.colors.length > 0) {
                    liquidColor = this.options.multicolor.colors[0];
                } else {
                    liquidColor = this.options.colors.circle || currentColor;
                }
            }
            
            let liquidRgb = this.hexToRgb(liquidColor);
            
            // CRITICAL: If liquidRgb is invalid or white, use a fallback
            if (!liquidRgb || (liquidRgb.r === 255 && liquidRgb.g === 255 && liquidRgb.b === 255)) {
                // Use circle color as fallback
                const fallbackColor = this.options.colors.circle || currentColor;
                const fallbackRgb = this.hexToRgb(fallbackColor);
                if (fallbackRgb) {
                    liquidRgb = fallbackRgb;
                } else {
                    // Last resort: use a default color
                    liquidRgb = { r: 100, g: 150, b: 255 };
                }
            }
            
            // Calculate target liquid level (inverse of progress - liquid decreases as time passes)
            // For countdown: progress goes from 1 to 0, so liquid level goes from full to empty
            // For countup: progress goes from 0 to 1, so liquid level goes from empty to full
            const unitKey = `${unit.label}-${index}`;
            const targetLiquidLevel = progress;
            
            // Initialize smooth liquid level if not exists
            if (this.liquidLevels[unitKey] === undefined) {
                this.liquidLevels[unitKey] = targetLiquidLevel;
            }
            if (this.liquidLevelTargets[unitKey] === undefined) {
                this.liquidLevelTargets[unitKey] = targetLiquidLevel;
            }
            
            // Smooth interpolation for fluid animation (natural, realistic liquid movement)
            this.liquidLevelTargets[unitKey] = targetLiquidLevel;
            const levelDiff = this.liquidLevelTargets[unitKey] - this.liquidLevels[unitKey];
            // Use adaptive easing: faster when far from target, slower when close (realistic liquid physics)
            const easingFactor = Math.abs(levelDiff) > 0.1 ? 0.06 : 0.04; // More responsive but still smooth
            this.liquidLevels[unitKey] += levelDiff * easingFactor;
            
            // Use smooth animated liquid level
            const liquidLevel = Math.max(0, Math.min(1, this.liquidLevels[unitKey]));
            
            // Calculate liquid height in circle (from bottom)
            const liquidHeight = circleRadius * 2 * liquidLevel;
            const liquidTop = circleY + circleRadius - liquidHeight;
            
            // Only draw if there's liquid
            if (liquidLevel > 0) {
                // Create clipping path for circle (3D container effect)
                ctx.save();
                ctx.beginPath();
                ctx.arc(circleX + parallaxOffsetX, circleY + parallaxOffsetY, circleRadius, 0, Math.PI * 2);
                ctx.clip();
                
                // 3D LIQUID EFFECT - Realistic depth and lighting
                // Draw liquid with radial gradient for 3D cylindrical container effect
                const centerX = circleX + parallaxOffsetX;
                const centerY = circleY + parallaxOffsetY;
                const liquidCenterY = (liquidTop + circleY + circleRadius + parallaxOffsetY) / 2;
                
                // Main liquid body with radial gradient (simulates 3D cylindrical container)
                const liquidRadialGradient = ctx.createRadialGradient(
                    centerX, liquidCenterY, 0, // Light source at center
                    centerX, liquidCenterY, circleRadius * 1.2 // Fades to edges
                );
                
                if (liquidRgb) {
                    // NATURAL 3D liquid - simple and realistic
                    // Simulates natural light and depth (not excessive)
                    liquidRadialGradient.addColorStop(0, `rgba(${liquidRgb.r}, ${liquidRgb.g}, ${liquidRgb.b}, 0.94)`); // Bright center
                    liquidRadialGradient.addColorStop(0.4, `rgba(${liquidRgb.r}, ${liquidRgb.g}, ${liquidRgb.b}, 0.90)`);
                    liquidRadialGradient.addColorStop(0.7, `rgba(${liquidRgb.r}, ${liquidRgb.g}, ${liquidRgb.b}, 0.85)`);
                    liquidRadialGradient.addColorStop(1, `rgba(${liquidRgb.r}, ${liquidRgb.g}, ${liquidRgb.b}, 0.80)`); // Slightly darker edges
                } else {
                    liquidRadialGradient.addColorStop(0, liquidColor);
                    liquidRadialGradient.addColorStop(1, liquidColor);
                }
                
                // Draw liquid body with 3D radial gradient - FOLLOWS WAVY SURFACE (no straight line visible)
                // Calculate wave parameters (same as in drawLiquidSurface for consistency)
                const waveSpeed = this.options.liquidEffect.waveSpeed || 1.0;
                const waveOffset = this.liquidWaveOffset || 0;
                const numWaves = 4;
                const waveAmplitude = 4.0;
                
                const startX = circleX - circleRadius + parallaxOffsetX;
                const endX = circleX + circleRadius + parallaxOffsetX;
                const steps = 180;
                const stepSize = (endX - startX) / steps;
                const bottomY = circleY + circleRadius + parallaxOffsetY;
                
                // Draw liquid body path that follows the wavy surface (no straight line visible)
                ctx.fillStyle = liquidRadialGradient;
                ctx.beginPath();
                ctx.moveTo(startX, bottomY); // Start from bottom left
                
                // Draw bottom edge
                ctx.lineTo(startX, bottomY);
                
                // Draw wavy top surface (follows the same wave pattern)
                for (let i = 0; i <= steps; i++) {
                    const x = startX + (i * stepSize);
                    const relativeX = (x - (circleX + parallaxOffsetX)) / circleRadius;
                    
                    // Same wave calculation as in drawLiquidSurface
                    const primaryWave = Math.sin((relativeX * Math.PI * numWaves) + waveOffset) * waveAmplitude;
                    const secondaryWave = Math.sin((relativeX * Math.PI * numWaves * 1.8) + waveOffset * 1.3) * (waveAmplitude * 0.55);
                    const detailWave = Math.sin((relativeX * Math.PI * numWaves * 3.2) + waveOffset * 2.2) * (waveAmplitude * 0.35);
                    const swellWave = Math.sin((relativeX * Math.PI * numWaves * 0.5) + waveOffset * 0.7) * (waveAmplitude * 0.4);
                    const waveY = liquidTop + parallaxOffsetY + primaryWave + secondaryWave + detailWave + swellWave;
                    
                    ctx.lineTo(x, waveY);
                }
                
                // Complete the path to bottom right and close
                ctx.lineTo(endX, bottomY);
                ctx.closePath();
                ctx.fill();
                
                // Add vertical gradient overlay for depth (darker at bottom, lighter at top) - FOLLOWS WAVY SURFACE
                const depthGradient = ctx.createLinearGradient(
                    centerX, circleY + circleRadius + parallaxOffsetY, // Bottom
                    centerX, liquidTop + parallaxOffsetY // Top
                );
                
                if (liquidRgb) {
                    // NATURAL: Simple depth effect (not excessive)
                    depthGradient.addColorStop(0, `rgba(0, 0, 0, 0.20)`); // Darker at bottom
                    depthGradient.addColorStop(0.5, `rgba(0, 0, 0, 0.10)`);
                    depthGradient.addColorStop(0.8, `rgba(0, 0, 0, 0.03)`);
                    depthGradient.addColorStop(1, `rgba(0, 0, 0, 0)`); // No shadow at top
                }
                
                // Draw depth gradient following wavy surface
                ctx.fillStyle = depthGradient;
                ctx.beginPath();
                ctx.moveTo(startX, bottomY);
                
                // Draw wavy top surface
                for (let i = 0; i <= steps; i++) {
                    const x = startX + (i * stepSize);
                    const relativeX = (x - (circleX + parallaxOffsetX)) / circleRadius;
                    const primaryWave = Math.sin((relativeX * Math.PI * numWaves) + waveOffset) * waveAmplitude;
                    const secondaryWave = Math.sin((relativeX * Math.PI * numWaves * 1.8) + waveOffset * 1.3) * (waveAmplitude * 0.55);
                    const detailWave = Math.sin((relativeX * Math.PI * numWaves * 3.2) + waveOffset * 2.2) * (waveAmplitude * 0.35);
                    const swellWave = Math.sin((relativeX * Math.PI * numWaves * 0.5) + waveOffset * 0.7) * (waveAmplitude * 0.4);
                    const waveY = liquidTop + parallaxOffsetY + primaryWave + secondaryWave + detailWave + swellWave;
                    ctx.lineTo(x, waveY);
                }
                
                ctx.lineTo(endX, bottomY);
                ctx.closePath();
                ctx.fill();
                
                // Add side shadows for 3D container walls effect
                const leftShadowGradient = ctx.createLinearGradient(
                    circleX - circleRadius + parallaxOffsetX, liquidCenterY,
                    circleX - circleRadius * 0.3 + parallaxOffsetX, liquidCenterY
                );
                leftShadowGradient.addColorStop(0, `rgba(0, 0, 0, 0.2)`); // Dark at left edge
                leftShadowGradient.addColorStop(1, `rgba(0, 0, 0, 0)`); // Fade to center
                
                ctx.fillStyle = leftShadowGradient;
                ctx.fillRect(
                    circleX - circleRadius + parallaxOffsetX,
                    liquidTop + parallaxOffsetY,
                    circleRadius * 0.6,
                    liquidHeight
                );
                
                const rightShadowGradient = ctx.createLinearGradient(
                    circleX + circleRadius * 0.3 + parallaxOffsetX, liquidCenterY,
                    circleX + circleRadius + parallaxOffsetX, liquidCenterY
                );
                rightShadowGradient.addColorStop(0, `rgba(0, 0, 0, 0)`); // Fade from center
                rightShadowGradient.addColorStop(1, `rgba(0, 0, 0, 0.2)`); // Dark at right edge
                
                ctx.fillStyle = rightShadowGradient;
                ctx.fillRect(
                    circleX + circleRadius * 0.3 + parallaxOffsetX,
                    liquidTop + parallaxOffsetY,
                    circleRadius * 0.7,
                    liquidHeight
                );
                
                // Add highlight from top (light source simulation) - FOLLOWS WAVY SURFACE
                const topHighlightGradient = ctx.createLinearGradient(
                    centerX, liquidTop + parallaxOffsetY,
                    centerX, liquidTop + liquidHeight * 0.3 + parallaxOffsetY
                );
                topHighlightGradient.addColorStop(0, `rgba(255, 255, 255, 0.15)`); // Bright highlight at top
                topHighlightGradient.addColorStop(0.5, `rgba(255, 255, 255, 0.08)`);
                topHighlightGradient.addColorStop(1, `rgba(255, 255, 255, 0)`); // Fade down
                
                // Draw top highlight following wavy surface (only top portion)
                ctx.fillStyle = topHighlightGradient;
                ctx.beginPath();
                const highlightStartX = circleX - circleRadius * 0.8 + parallaxOffsetX;
                const highlightEndX = circleX + circleRadius * 0.8 + parallaxOffsetX;
                const highlightSteps = Math.floor(steps * 0.8);
                const highlightStepSize = (highlightEndX - highlightStartX) / highlightSteps;
                
                // Start from left edge of highlight area
                let highlightStartY = liquidTop + parallaxOffsetY;
                if (highlightStartX >= startX) {
                    const relativeX = (highlightStartX - (circleX + parallaxOffsetX)) / circleRadius;
                    const primaryWave = Math.sin((relativeX * Math.PI * numWaves) + waveOffset) * waveAmplitude;
                    const secondaryWave = Math.sin((relativeX * Math.PI * numWaves * 1.8) + waveOffset * 1.3) * (waveAmplitude * 0.55);
                    const detailWave = Math.sin((relativeX * Math.PI * numWaves * 3.2) + waveOffset * 2.2) * (waveAmplitude * 0.35);
                    const swellWave = Math.sin((relativeX * Math.PI * numWaves * 0.5) + waveOffset * 0.7) * (waveAmplitude * 0.4);
                    highlightStartY = liquidTop + parallaxOffsetY + primaryWave + secondaryWave + detailWave + swellWave;
                }
                ctx.moveTo(highlightStartX, highlightStartY);
                
                // Draw wavy top edge
                for (let i = 0; i <= highlightSteps; i++) {
                    const x = highlightStartX + (i * highlightStepSize);
                    const relativeX = (x - (circleX + parallaxOffsetX)) / circleRadius;
                    const primaryWave = Math.sin((relativeX * Math.PI * numWaves) + waveOffset) * waveAmplitude;
                    const secondaryWave = Math.sin((relativeX * Math.PI * numWaves * 1.8) + waveOffset * 1.3) * (waveAmplitude * 0.55);
                    const detailWave = Math.sin((relativeX * Math.PI * numWaves * 3.2) + waveOffset * 2.2) * (waveAmplitude * 0.35);
                    const swellWave = Math.sin((relativeX * Math.PI * numWaves * 0.5) + waveOffset * 0.7) * (waveAmplitude * 0.4);
                    const waveY = liquidTop + parallaxOffsetY + primaryWave + secondaryWave + detailWave + swellWave;
                    ctx.lineTo(x, waveY);
                }
                
                // Draw bottom edge (straight, but follows wave pattern at top)
                const highlightBottomY = liquidTop + liquidHeight * 0.3 + parallaxOffsetY;
                ctx.lineTo(highlightEndX, highlightBottomY);
                ctx.lineTo(highlightStartX, highlightBottomY);
                ctx.closePath();
                ctx.fill();
                
                // Draw liquid surface with wave effect
                this.drawLiquidSurface(ctx, circleX, circleY, circleRadius, liquidTop, liquidRgb, parallaxOffsetX, parallaxOffsetY);
                
                // Update and draw bubbles
                this.updateLiquidBubbles(unit, index, circleX, circleY, circleRadius, liquidTop, liquidLevel);
                this.drawLiquidBubbles(ctx, unit, index, circleX, circleY, circleRadius, liquidTop, liquidRgb, parallaxOffsetX, parallaxOffsetY);
                
                // Restore clipping (for liquid fill only)
                ctx.restore();
                
                // Draw evaporation effect at surface if enabled (OUTSIDE the circle - no clipping)
                if (this.options.liquidEffect.showEvaporation && liquidLevel > 0.05) {
                    this.drawEvaporationEffect(ctx, circleX, circleY, circleRadius, liquidTop, liquidRgb, parallaxOffsetX, parallaxOffsetY);
                }
            }
            
            ctx.restore();
            
            // Draw 3D container border with inner shadow for depth
            // CRITICAL: Border uses circleBackground (separate from circle color)
            // Apply multicolor, pulseGlow, and gradientWave effects to the border
            ctx.save();
            ctx.beginPath();
            ctx.arc(circleX + parallaxOffsetX, circleY + parallaxOffsetY, circleRadius, 0, Math.PI * 2);
            
            // Use circleBackground as base color (separate from circle color)
            const borderBaseColor = this.options.colors.circleBackground;
            
            // Get colors for border effects - use multicolor if enabled, otherwise use circleBackground
            let borderCurrentColor = borderBaseColor;
            let borderColors = [borderBaseColor];
            
            if (this.options.multicolor && this.options.multicolor.enabled) {
                // Use multicolor colors for border
                borderColors = this.getCurrentColors();
                borderCurrentColor = borderColors[index % borderColors.length];
            }
            
            // Create gradient for border (multicolor support)
            const borderGradient = ctx.createLinearGradient(
                circleX - circleRadius + parallaxOffsetX,
                circleY - circleRadius + parallaxOffsetY,
                circleX + circleRadius + parallaxOffsetX,
                circleY + circleRadius + parallaxOffsetY
            );
            
            if (this.options.multicolor && this.options.multicolor.enabled && borderColors.length > 1) {
                const color1 = this.hexToRgb(borderCurrentColor);
                const nextColor = borderColors[(index + 1) % borderColors.length];
                const color2 = this.hexToRgb(nextColor);
                
                if (color1 && color2) {
                    borderGradient.addColorStop(0, `rgba(${color1.r}, ${color1.g}, ${color1.b}, 1)`);
                    borderGradient.addColorStop(1, `rgba(${color2.r}, ${color2.g}, ${color2.b}, 1)`);
                } else {
                    borderGradient.addColorStop(0, borderCurrentColor);
                    borderGradient.addColorStop(1, borderCurrentColor);
                }
            } else {
                borderGradient.addColorStop(0, borderBaseColor);
                borderGradient.addColorStop(1, borderBaseColor);
            }
            
            // Apply Gradient Wave animation to border (only if multicolor is disabled)
            if (this.options.gradientWave && this.options.gradientWave.enabled && 
                (!this.options.multicolor || !this.options.multicolor.enabled)) {
                const waveColors = this.options.gradientWave.colors || ['#667eea', '#764ba2', '#f093fb', '#4facfe'];
                const waveOffset = this.gradientWaveProgress;
                const numColors = waveColors.length;
                
                // Create animated gradient for border
                const waveGradient = ctx.createLinearGradient(
                    circleX - circleRadius + parallaxOffsetX,
                    circleY - circleRadius + parallaxOffsetY,
                    circleX + circleRadius + parallaxOffsetX,
                    circleY + circleRadius + parallaxOffsetY
                );
                
                for (let i = 0; i < numColors; i++) {
                    const stop = (i / numColors + waveOffset) % 1;
                    waveGradient.addColorStop(stop, waveColors[i]);
                }
                
                ctx.strokeStyle = waveGradient;
            } else {
                ctx.strokeStyle = borderGradient;
            }
            
            // Apply Pulse/Glow animation to border
            // Reduce border thickness on mobile for better proportions
            const isMobileBorder = circleRadius < 35; // Mobile circles are smaller
            const baseBorderThickness = isMobileBorder 
                ? Math.max(2, Math.min(4, this.options.circleThickness * 0.3)) // 30% of original, min 2px, max 4px
                : this.options.circleThickness;
            let borderLineWidth = baseBorderThickness;
            let borderShadowBlur = 15;
            
            // CRITICAL: Ensure pulseGlow exists and is properly initialized before checking enabled
            if (!this.options.pulseGlow) {
                this.options.pulseGlow = {
                    enabled: false,
                    intensity: 1.0,
                    speed: 1.0
                };
            }
            
            if (this.options.pulseGlow && this.options.pulseGlow.enabled) {
                const pulseIntensity = (Math.sin(this.pulseProgress) + 1) / 2; // 0 to 1
                const pulseIntensityValue = this.options.pulseGlow.intensity || 1.0;
                borderShadowBlur = 15 + (pulseIntensity * 20 * pulseIntensityValue);
                borderLineWidth = baseBorderThickness + (pulseIntensity * 2 * pulseIntensityValue);
                
                // Apply glow shadow using border color
                ctx.shadowColor = borderCurrentColor;
                ctx.shadowBlur = borderShadowBlur;
                ctx.shadowOffsetX = 0;
                ctx.shadowOffsetY = 0;
            } else {
                ctx.shadowColor = 'rgba(0, 0, 0, 0)';
                ctx.shadowBlur = 0;
                ctx.shadowOffsetX = 0;
                ctx.shadowOffsetY = 0;
            }
            
            ctx.lineWidth = borderLineWidth;
            ctx.lineJoin = 'round';
            ctx.lineCap = 'round';
            ctx.stroke();
            
            // Inner shadow for 3D container depth effect (gives impression of thickness)
            ctx.beginPath();
            ctx.arc(circleX + parallaxOffsetX, circleY + parallaxOffsetY, circleRadius - 1, 0, Math.PI * 2);
            ctx.strokeStyle = `rgba(0, 0, 0, 0.15)`; // Subtle inner shadow for 3D depth
            ctx.lineWidth = 1;
            ctx.stroke();
            
            ctx.restore();
        }
        
        /**
         * Draw liquid surface with wave animation
         */
        drawLiquidSurface(ctx, circleX, circleY, circleRadius, liquidTop, liquidRgb, parallaxOffsetX, parallaxOffsetY) {
            if (!liquidRgb) return;
            
            // CRITICAL: Ensure liquidWaveOffset is always defined and valid - prevents waves disappearing
            // This check must happen FIRST, before any other operations
            if (this.liquidWaveOffset === undefined || 
                this.liquidWaveOffset === null || 
                isNaN(this.liquidWaveOffset) ||
                !isFinite(this.liquidWaveOffset)) {
                // Only reset if truly invalid - preserve existing value otherwise
                // Check if we have a preserved value from options
                if (this.options._preservedLiquidWaveOffset !== undefined && 
                    !isNaN(this.options._preservedLiquidWaveOffset) && 
                    isFinite(this.options._preservedLiquidWaveOffset)) {
                    this.liquidWaveOffset = this.options._preservedLiquidWaveOffset;
                } else {
                    this.liquidWaveOffset = 0;
                }
            }
            
            // CRITICAL: Ensure liquidEffect exists and is enabled
            if (!this.options.liquidEffect) {
                this.options.liquidEffect = {
                    enabled: true,
                    intensity: 1.2,
                    bubbleSpeed: 1.0,
                    waveSpeed: 1.0,
                    bubbleSpawnRate: 0.8,
                    showEvaporation: true,
                    liquidColor: null
                };
            }
            this.options.liquidEffect.enabled = true;
            
            const waveSpeed = this.options.liquidEffect.waveSpeed || 1.0;
            // CRITICAL: Use current liquidWaveOffset value - it's continuously updated in animate()
            // Never reset or recalculate - just use the existing value
            const waveOffset = this.liquidWaveOffset || 0; // Use existing offset, fallback to 0 only if truly missing
            
            ctx.save();
            
            // Create wavy surface using multiple sine waves (ULTRA-REALISTIC NATURAL LIQUID)
            // Realistic liquid waves: moderate amplitude, natural frequencies, smooth motion
            const numWaves = 4; // Optimal number of waves for realistic liquid surface
            const waveAmplitude = 4.0; // Natural amplitude for realistic liquid movement
            
            // Start from left edge of circle
            const startX = circleX - circleRadius + parallaxOffsetX;
            const endX = circleX + circleRadius + parallaxOffsetX;
            const steps = 180; // Optimal steps for smooth, realistic liquid surface
            const stepSize = (endX - startX) / steps;
            
            // Draw wavy surface path with natural wave frequencies for ultra-realistic liquid
            ctx.beginPath();
            ctx.moveTo(startX, liquidTop + parallaxOffsetY);
            
            for (let i = 0; i <= steps; i++) {
                const x = startX + (i * stepSize);
                const relativeX = (x - (circleX + parallaxOffsetX)) / circleRadius;
                
                // Natural liquid waves: combination of primary wave, secondary ripples, and subtle details
                // Each wave moves at natural speeds to create organic, realistic liquid motion
                const primaryWave = Math.sin((relativeX * Math.PI * numWaves) + waveOffset) * waveAmplitude;
                const secondaryWave = Math.sin((relativeX * Math.PI * numWaves * 1.8) + waveOffset * 1.3) * (waveAmplitude * 0.55); // Natural secondary ripple
                const detailWave = Math.sin((relativeX * Math.PI * numWaves * 3.2) + waveOffset * 2.2) * (waveAmplitude * 0.35); // Subtle surface detail
                const swellWave = Math.sin((relativeX * Math.PI * numWaves * 0.5) + waveOffset * 0.7) * (waveAmplitude * 0.4); // Gentle swell
                
                // Combine waves naturally for realistic liquid surface
                const y = liquidTop + parallaxOffsetY + primaryWave + secondaryWave + detailWave + swellWave;
                ctx.lineTo(x, y);
            }
            
            // Complete the path to bottom of circle and close
            ctx.lineTo(endX, circleY + circleRadius + parallaxOffsetY);
            ctx.lineTo(startX, circleY + circleRadius + parallaxOffsetY);
            ctx.closePath();
            
            // NATURAL LIQUID SURFACE - Simple and realistic
            // Fill with natural liquid gradient (simple and clean)
            const surfaceGradient = ctx.createLinearGradient(
                circleX + parallaxOffsetX, liquidTop + parallaxOffsetY - 10,
                circleX + parallaxOffsetX, liquidTop + parallaxOffsetY + 5
            );
            // Natural liquid colors - simple gradient
            surfaceGradient.addColorStop(0, `rgba(${liquidRgb.r}, ${liquidRgb.g}, ${liquidRgb.b}, 0.96)`); // Bright at top
            surfaceGradient.addColorStop(0.5, `rgba(${liquidRgb.r}, ${liquidRgb.g}, ${liquidRgb.b}, 0.93)`);
            surfaceGradient.addColorStop(1, `rgba(${liquidRgb.r}, ${liquidRgb.g}, ${liquidRgb.b}, 0.90)`); // Slightly darker below
            ctx.fillStyle = surfaceGradient;
            ctx.fill();
            
            // 3D SURFACE HIGHLIGHTS - Realistic reflections and depth (NATURAL WAVES)
            // Main highlight (bright reflection from light source above) - follows natural waves
            ctx.beginPath();
            ctx.moveTo(startX, liquidTop + parallaxOffsetY);
            for (let i = 0; i <= steps; i++) {
                const x = startX + (i * stepSize);
                const relativeX = (x - (circleX + parallaxOffsetX)) / circleRadius;
                const primaryWave = Math.sin((relativeX * Math.PI * numWaves) + waveOffset) * waveAmplitude;
                const secondaryWave = Math.sin((relativeX * Math.PI * numWaves * 1.8) + waveOffset * 1.3) * (waveAmplitude * 0.55);
                const detailWave = Math.sin((relativeX * Math.PI * numWaves * 3.2) + waveOffset * 2.2) * (waveAmplitude * 0.35);
                const swellWave = Math.sin((relativeX * Math.PI * numWaves * 0.5) + waveOffset * 0.7) * (waveAmplitude * 0.4);
                const y = liquidTop + parallaxOffsetY + primaryWave + secondaryWave + detailWave + swellWave;
                ctx.lineTo(x, y);
            }
            
            // NATURAL: Simple highlight (realistic but not excessive)
            const highlightGradient = ctx.createLinearGradient(
                circleX - circleRadius * 0.4 + parallaxOffsetX, liquidTop + parallaxOffsetY,
                circleX + circleRadius * 0.4 + parallaxOffsetX, liquidTop + parallaxOffsetY
            );
            highlightGradient.addColorStop(0, `rgba(255, 255, 255, 0.3)`); // Softer at edges
            highlightGradient.addColorStop(0.5, `rgba(255, 255, 255, 0.7)`); // Bright at center
            highlightGradient.addColorStop(1, `rgba(255, 255, 255, 0.3)`); // Softer at edges
            
            ctx.strokeStyle = highlightGradient;
            ctx.lineWidth = 2;
            ctx.lineCap = 'round';
            ctx.stroke();
            
            // Secondary highlight (softer reflection below main) - follows natural waves
            ctx.beginPath();
            ctx.moveTo(startX, liquidTop + parallaxOffsetY);
            for (let i = 0; i <= steps; i++) {
                const x = startX + (i * stepSize);
                const relativeX = (x - (circleX + parallaxOffsetX)) / circleRadius;
                const primaryWave = Math.sin((relativeX * Math.PI * numWaves) + waveOffset) * waveAmplitude;
                const secondaryWave = Math.sin((relativeX * Math.PI * numWaves * 1.8) + waveOffset * 1.3) * (waveAmplitude * 0.55);
                const detailWave = Math.sin((relativeX * Math.PI * numWaves * 3.2) + waveOffset * 2.2) * (waveAmplitude * 0.35);
                const swellWave = Math.sin((relativeX * Math.PI * numWaves * 0.5) + waveOffset * 0.7) * (waveAmplitude * 0.4);
                const y = liquidTop + parallaxOffsetY + primaryWave + secondaryWave + detailWave + swellWave - 1.5; // Below main highlight
                ctx.lineTo(x, y);
            }
            ctx.strokeStyle = `rgba(255, 255, 255, 0.25)`; // Softer secondary highlight
            ctx.lineWidth = 1.5;
            ctx.stroke();
            
            // 3D depth shadow below surface (realistic liquid depth effect) - follows natural waves
            ctx.beginPath();
            ctx.moveTo(startX, liquidTop + parallaxOffsetY);
            for (let i = 0; i <= steps; i++) {
                const x = startX + (i * stepSize);
                const relativeX = (x - (circleX + parallaxOffsetX)) / circleRadius;
                const primaryWave = Math.sin((relativeX * Math.PI * numWaves) + waveOffset) * waveAmplitude;
                const secondaryWave = Math.sin((relativeX * Math.PI * numWaves * 1.8) + waveOffset * 1.3) * (waveAmplitude * 0.55);
                const detailWave = Math.sin((relativeX * Math.PI * numWaves * 3.2) + waveOffset * 2.2) * (waveAmplitude * 0.35);
                const swellWave = Math.sin((relativeX * Math.PI * numWaves * 0.5) + waveOffset * 0.7) * (waveAmplitude * 0.4);
                const y = liquidTop + parallaxOffsetY + primaryWave + secondaryWave + detailWave + swellWave + 3; // Below surface
                ctx.lineTo(x, y);
            }
            ctx.strokeStyle = `rgba(0, 0, 0, 0.2)`; // Stronger shadow for 3D depth
            ctx.lineWidth = 2;
            ctx.stroke();
            
            // Additional depth shadow (very subtle, for extra 3D effect) - follows natural waves
            ctx.beginPath();
            ctx.moveTo(startX, liquidTop + parallaxOffsetY);
            for (let i = 0; i <= steps; i++) {
                const x = startX + (i * stepSize);
                const relativeX = (x - (circleX + parallaxOffsetX)) / circleRadius;
                const primaryWave = Math.sin((relativeX * Math.PI * numWaves) + waveOffset) * waveAmplitude;
                const secondaryWave = Math.sin((relativeX * Math.PI * numWaves * 1.8) + waveOffset * 1.3) * (waveAmplitude * 0.55);
                const detailWave = Math.sin((relativeX * Math.PI * numWaves * 3.2) + waveOffset * 2.2) * (waveAmplitude * 0.35);
                const swellWave = Math.sin((relativeX * Math.PI * numWaves * 0.5) + waveOffset * 0.7) * (waveAmplitude * 0.4);
                const y = liquidTop + parallaxOffsetY + primaryWave + secondaryWave + detailWave + swellWave + 5; // Even deeper
                ctx.lineTo(x, y);
            }
            ctx.strokeStyle = `rgba(0, 0, 0, 0.1)`; // Subtle additional shadow
            ctx.lineWidth = 1;
            ctx.stroke();
            
            ctx.restore();
        }
        
        /**
         * Update liquid bubbles (spawn and move)
         */
        updateLiquidBubbles(unit, index, circleX, circleY, circleRadius, liquidTop, liquidLevel) {
            const unitKey = `${index}_${unit.label}`;
            
            // Initialize bubbles array for this unit if needed
            if (!this.liquidBubbles[unitKey]) {
                this.liquidBubbles[unitKey] = [];
                this.lastBubbleSpawn[unitKey] = Date.now();
            }
            
            const bubbles = this.liquidBubbles[unitKey];
            const bubbleSpeed = this.options.liquidEffect.bubbleSpeed || 1.0;
            const spawnRate = this.options.liquidEffect.bubbleSpawnRate || 0.5;
            const intensity = this.options.liquidEffect.intensity || 1.0;
            
            const now = Date.now();
            const timeSinceLastSpawn = now - this.lastBubbleSpawn[unitKey];
            // Realistic spawn interval for still liquid (slower, more natural)
            const spawnInterval = (5000 / spawnRate) / intensity; // 5 seconds base (much slower for still liquid)
            
            // Spawn new bubbles (REALISTIC - slow, natural movement in still liquid)
            if (liquidLevel > 0.1 && timeSinceLastSpawn > spawnInterval) {
                // Spawn 1-2 bubbles occasionally (realistic for still liquid)
                const numBubbles = Math.random() > 0.6 ? 2 : 1; // Mostly 1 bubble, sometimes 2
                for (let i = 0; i < numBubbles; i++) {
                    const bubbleX = circleX + (Math.random() - 0.5) * circleRadius * 1.2;
                    // Spawn from bottom 30% of liquid (more realistic)
                    const bubbleY = liquidTop + Math.random() * (circleRadius * 2 * liquidLevel * 0.3) + (circleRadius * 2 * liquidLevel * 0.7);
                    // Realistic bubble sizes: mostly small, occasionally medium
                    const sizeRandom = Math.random();
                    const bubbleSize = sizeRandom > 0.7 ? (Math.random() * 1.5 + 2) : (Math.random() * 1 + 1); // 1-2px (small), 2-3.5px (medium)
                    // Very slow speed for realistic still liquid (0.05-0.15 * speed)
                    const bubbleSpeedValue = (Math.random() * 0.1 + 0.05) * bubbleSpeed * 0.2; // Much slower
                    
                    bubbles.push({
                        x: bubbleX,
                        y: bubbleY,
                        size: bubbleSize,
                        speed: bubbleSpeedValue,
                        opacity: Math.random() * 0.4 + 0.4, // 0.4-0.8 (more visible)
                        wobble: Math.random() * Math.PI * 2, // Random wobble phase
                        wobbleSpeed: Math.random() * 0.02 + 0.01, // Very slow wobble (realistic)
                        wobbleAmplitude: Math.random() * 0.3 + 0.2 // Small lateral movement
                    });
                }
                this.lastBubbleSpawn[unitKey] = now;
            }
            
            // Update existing bubbles (REALISTIC - slow, natural movement)
            const activeBubbles = [];
            for (let i = 0; i < bubbles.length; i++) {
                const bubble = bubbles[i];
                
                // Move bubble upward (very slowly, like in still liquid)
                bubble.y -= bubble.speed;
                
                // Add subtle horizontal wobble (realistic for still liquid)
                bubble.wobble += bubble.wobbleSpeed;
                const wobbleAmplitude = bubble.wobbleAmplitude || 0.25;
                bubble.x += Math.sin(bubble.wobble) * wobbleAmplitude; // Very subtle lateral movement
                
                // Slight size variation as bubble rises (realistic expansion)
                const distanceFromStart = (bubble.y - (liquidTop + circleRadius * 2 * liquidLevel * 0.7)) / (circleRadius * 2 * liquidLevel);
                if (distanceFromStart > 0) {
                    bubble.size = bubble.size * (1 + distanceFromStart * 0.1); // Slight expansion
                }
                
                // Fade out gradually as bubble approaches surface (realistic)
                const distanceFromSurface = liquidTop - bubble.y;
                const maxDistance = circleRadius * 2;
                if (distanceFromSurface > 0 && distanceFromSurface < maxDistance * 0.3) {
                    // Only fade in last 30% of journey
                    bubble.opacity = Math.max(0.2, bubble.opacity - (distanceFromSurface / (maxDistance * 0.3)) * 0.01);
                }
                
                // Keep bubble if it's below surface and not faded out
                if (bubble.y < liquidTop + 5 && bubble.opacity > 0.15) {
                    activeBubbles.push(bubble);
                }
            }
            
            this.liquidBubbles[unitKey] = activeBubbles;
        }
        
        /**
         * Draw liquid bubbles
         */
        drawLiquidBubbles(ctx, unit, index, circleX, circleY, circleRadius, liquidTop, liquidRgb, parallaxOffsetX, parallaxOffsetY) {
            const unitKey = `${index}_${unit.label}`;
            const bubbles = this.liquidBubbles[unitKey] || [];
            
            if (!liquidRgb) return;
            
            ctx.save();
            
            for (let i = 0; i < bubbles.length; i++) {
                const bubble = bubbles[i];
                
                // Only draw bubbles below surface
                if (bubble.y < liquidTop) {
                    ctx.globalAlpha = bubble.opacity;
                    
                    // Realistic bubble with natural gradient (like real bubbles in still liquid)
                    const bubbleGradient = ctx.createRadialGradient(
                        bubble.x + parallaxOffsetX - bubble.size * 0.25,
                        bubble.y + parallaxOffsetY - bubble.size * 0.25,
                        0,
                        bubble.x + parallaxOffsetX,
                        bubble.y + parallaxOffsetY,
                        bubble.size
                    );
                    // More realistic bubble appearance (less bright, more transparent)
                    bubbleGradient.addColorStop(0, `rgba(255, 255, 255, 0.7)`); // Softer highlight
                    bubbleGradient.addColorStop(0.4, `rgba(255, 255, 255, 0.5)`);
                    bubbleGradient.addColorStop(0.7, `rgba(255, 255, 255, 0.3)`);
                    bubbleGradient.addColorStop(1, `rgba(255, 255, 255, 0.15)`); // Very transparent edges
                    
                    ctx.fillStyle = bubbleGradient;
                    ctx.beginPath();
                    ctx.arc(
                        bubble.x + parallaxOffsetX,
                        bubble.y + parallaxOffsetY,
                        bubble.size,
                        0,
                        Math.PI * 2
                    );
                    ctx.fill();
                    
                    // Subtle highlight spot on bubble (realistic, not too bright)
                    ctx.fillStyle = `rgba(255, 255, 255, 0.6)`;
                    ctx.beginPath();
                    ctx.arc(
                        bubble.x + parallaxOffsetX - bubble.size * 0.2,
                        bubble.y + parallaxOffsetY - bubble.size * 0.2,
                        bubble.size * 0.3,
                        0,
                        Math.PI * 2
                    );
                    ctx.fill();
                    
                    // Very subtle border for realism (optional, very light)
                    if (bubble.size > 1.5) {
                        ctx.strokeStyle = `rgba(255, 255, 255, 0.2)`;
                        ctx.lineWidth = 0.5;
                        ctx.beginPath();
                        ctx.arc(
                            bubble.x + parallaxOffsetX,
                            bubble.y + parallaxOffsetY,
                            bubble.size,
                            0,
                            Math.PI * 2
                        );
                        ctx.stroke();
                    }
                }
            }
            
            ctx.restore();
        }
        
        /**
         * Draw evaporation effect at liquid surface (INSIDE and OUTSIDE the circle)
         */
        drawEvaporationEffect(ctx, circleX, circleY, circleRadius, liquidTop, liquidRgb, parallaxOffsetX, parallaxOffsetY) {
            if (!liquidRgb) return;
            
            const unitKey = `${circleX}-${circleY}`;
            
            // Initialize evaporation particles if not exists
            if (!this.evaporationParticles[unitKey]) {
                this.evaporationParticles[unitKey] = [];
            }
            
            const particles = this.evaporationParticles[unitKey];
            const waveOffset = this.liquidWaveOffset || 0;
            const now = Date.now();
            
            // Spawn new particles at surface (both inside and outside circle)
            if (Math.random() < 0.3) { // 30% chance per frame
                const angle = Math.random() * Math.PI * 2;
                const spawnRadius = circleRadius * (0.7 + Math.random() * 0.3); // Spawn near edge
                const x = circleX + parallaxOffsetX + Math.cos(angle) * spawnRadius;
                const y = liquidTop + parallaxOffsetY + Math.sin(angle) * (circleRadius * 0.1);
                
                particles.push({
                    x: x,
                    y: y,
                    startX: x,
                    startY: y,
                    vx: (Math.random() - 0.5) * 0.5, // Horizontal drift
                    vy: -(Math.random() * 1.5 + 0.5), // Upward velocity
                    size: Math.random() * 2 + 1,
                    opacity: Math.random() * 0.4 + 0.3,
                    life: 1.0, // Life starts at 1.0, decreases to 0
                    spawnTime: now
                });
            }
            
            // Update and draw particles
            ctx.save();
            for (let i = particles.length - 1; i >= 0; i--) {
                const particle = particles[i];
                
                // Update position
                particle.x += particle.vx;
                particle.y += particle.vy;
                
                // Fade out over time
                particle.life -= 0.02; // Decrease life
                particle.opacity = particle.life * 0.5;
                
                // Remove dead particles
                if (particle.life <= 0 || particle.y < liquidTop - circleRadius * 2) {
                    particles.splice(i, 1);
                    continue;
                }
                
                // Draw particle (can be inside or outside circle)
                ctx.globalAlpha = particle.opacity;
                
                // High definition vapor with gradient
                const vaporGradient = ctx.createRadialGradient(
                    particle.x, particle.y, 0,
                    particle.x, particle.y, particle.size
                );
                vaporGradient.addColorStop(0, `rgba(${liquidRgb.r}, ${liquidRgb.g}, ${liquidRgb.b}, 0.8)`);
                vaporGradient.addColorStop(0.5, `rgba(${liquidRgb.r}, ${liquidRgb.g}, ${liquidRgb.b}, 0.4)`);
                vaporGradient.addColorStop(1, `rgba(${liquidRgb.r}, ${liquidRgb.g}, ${liquidRgb.b}, 0)`);
                
                ctx.fillStyle = vaporGradient;
                ctx.beginPath();
                ctx.arc(particle.x, particle.y, particle.size, 0, Math.PI * 2);
                ctx.fill();
            }
            
            // Limit particles count for performance
            if (particles.length > 50) {
                particles.splice(0, particles.length - 50);
            }
            
            ctx.restore();
        }
        
        /**
         * Easing function for smooth animations
         */
        easeInOutQuad(t) {
            return t < 0.5 ? 2 * t * t : -1 + (4 - 2 * t) * t;
        }

        /**
         * Get current colors based on multicolor settings
         */
        getCurrentColors() {
            if (this.options.multicolor.enabled && this.options.multicolor.colors.length > 0) {
                const colors = this.options.multicolor.colors;
                
                // If only one color, return it
                if (colors.length === 1) {
                    return [colors[0]];
                }
                
                // For multiple colors, cycle through them with smooth transitions
                const currentIndex = this.colorIndex;
                const nextIndex = (currentIndex + 1) % colors.length;
                const progress = this.animationProgress;
                
                // Return array of colors, interpolating the current transitioning ones
                return colors.map((color, index) => {
                    if (index === currentIndex) {
                        return this.interpolateColor(
                            colors[currentIndex],
                            colors[nextIndex],
                            progress
                        );
                    } else if (index === nextIndex) {
                        return this.interpolateColor(
                            colors[nextIndex],
                            colors[(nextIndex + 1) % colors.length],
                            progress
                        );
                    }
                    return color;
                });
            }
            
            // Single color mode
            return [this.options.colors.circle];
        }

        /**
         * Interpolate between two colors
         */
        interpolateColor(color1, color2, progress) {
            const c1 = this.hexToRgb(color1);
            const c2 = this.hexToRgb(color2);
            
            if (!c1 || !c2) return color1;
            
            const r = Math.round(c1.r + (c2.r - c1.r) * progress);
            const g = Math.round(c1.g + (c2.g - c1.g) * progress);
            const b = Math.round(c1.b + (c2.b - c1.b) * progress);
            
            return `rgb(${r}, ${g}, ${b})`;
        }

        /**
         * Convert hex color to RGB
         */
        hexToRgb(hex) {
            const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
            return result ? {
                r: parseInt(result[1], 16),
                g: parseInt(result[2], 16),
                b: parseInt(result[3], 16)
            } : null;
        }

        /**
         * Setup auto-reset functionality
         */
        setupAutoReset() {
            const interval = this.getAutoResetInterval();
            if (interval > 0) {
                setInterval(() => {
                    this.reset();
                }, interval);
            }
        }

        /**
         * Get auto-reset interval in milliseconds
         */
        getAutoResetInterval() {
            const unit = this.options.autoReset.unit;
            const value = this.options.autoReset.value;
            
            switch(unit) {
                case 'minutes':
                    return value * 60 * 1000;
                case 'hours':
                    return value * 60 * 60 * 1000;
                case 'days':
                    return value * 24 * 60 * 60 * 1000;
                default:
                    return 0;
            }
        }

        /**
         * Handle timer finish
         */
        onFinish() {
            this.stop();
            
            // Play finish sound if enabled
            if (this.options.sound.enabled && this.options.sound.finishEnabled) {
                this.playFinishSound();
            }
            
            // Start celebration animation if enabled
            if (this.options.celebration.enabled) {
                this.startCelebration();
            }
            
            // Trigger finish callback
            if (typeof this.options.onFinish === 'function') {
                this.options.onFinish.call(this, this);
            }
            
            // Handle expire action (with delay if celebration is enabled)
            const delay = this.options.celebration.enabled ? this.options.celebration.duration : 0;
            
            setTimeout(() => {
                switch(this.options.expireAction) {
                    case 'hide':
                        this.$element.fadeOut(500);
                        break;
                    case 'redirect':
                        if (this.options.expireRedirectUrl) {
                            window.location.href = this.options.expireRedirectUrl;
                        }
                        break;
                    case 'message':
                        if (this.options.expireMessage) {
                            this.$element.html(
                                '<div class="countdown-timer-pro-message">' + 
                                this.options.expireMessage + 
                                '</div>'
                            );
                        }
                        break;
                }
            }, delay);
        }

        /**
         * Start celebration animation (confetti/particles)
         */
        startCelebration() {
            if (this.isCelebrating) return;
            
            this.isCelebrating = true;
            
            // Create celebration canvas overlay
            this.createCelebrationCanvas();
            
            // Generate particles based on animation type
            const dpr = window.devicePixelRatio || 1;
            const width = this.canvas.width / dpr;
            const height = this.canvas.height / dpr;
            const centerX = width / 2;
            const centerY = height / 2;
            
            this.celebrationParticles = [];
            
            if (this.options.celebration.type === 'confetti' || this.options.celebration.type === 'both') {
                // Create confetti particles
                const confettiCount = this.options.celebration.intensity * 50;
                for (let i = 0; i < confettiCount; i++) {
                    this.celebrationParticles.push(this.createConfettiParticle(centerX, centerY, width, height));
                }
            }
            
            if (this.options.celebration.type === 'particles' || this.options.celebration.type === 'both') {
                // Create particle explosion
                const particleCount = this.options.celebration.intensity * 30;
                for (let i = 0; i < particleCount; i++) {
                    this.celebrationParticles.push(this.createParticle(centerX, centerY));
                }
            }
            
            // Start animation loop
            this.animateCelebration();
        }
        
        /**
         * Create celebration canvas overlay
         */
        createCelebrationCanvas() {
            // Ensure element has position relative for absolute positioning
            const elementStyle = window.getComputedStyle(this.element);
            if (elementStyle.position === 'static') {
                this.$element.css('position', 'relative');
            }
            
            // Remove existing canvas if present
            if (this.celebrationCanvas) {
                $(this.celebrationCanvas).remove();
                this.celebrationCanvas = null;
                this.celebrationCtx = null;
            }
            
            const dpr = window.devicePixelRatio || 1;
            const width = this.canvas.width;
            const height = this.canvas.height;
            
            this.celebrationCanvas = document.createElement('canvas');
            this.celebrationCanvas.className = 'countdown-timer-pro-celebration';
            this.celebrationCanvas.width = width;
            this.celebrationCanvas.height = height;
            
            // Position canvas to match timer canvas exactly - use same positioning as timer canvas
            this.celebrationCanvas.style.cssText = `
                position: absolute;
                top: 0;
                left: 0;
                width: ${width / dpr}px;
                height: ${height / dpr}px;
                pointer-events: none;
                z-index: 1000;
            `;
            
            this.celebrationCtx = this.celebrationCanvas.getContext('2d');
            this.celebrationCtx.scale(dpr, dpr);
            
            // Insert after timer canvas (same parent)
            if (this.canvas.parentNode) {
                this.canvas.parentNode.appendChild(this.celebrationCanvas);
            } else {
                this.$element.append(this.celebrationCanvas);
            }
        }
        
        /**
         * Create a confetti particle
         */
        createConfettiParticle(centerX, centerY, width, height) {
            const colors = this.options.celebration.colors || ['#667eea', '#764ba2', '#f093fb', '#4facfe', '#43e97b', '#fa709a', '#fee140', '#30cfd0'];
            const angle = (Math.PI * 2 * Math.random());
            // Confetti: slower, more colorful, bigger pieces
            const velocity = 1.5 + Math.random() * 3;
            const size = 8 + Math.random() * 12; // Bigger confetti pieces
            
            return {
                type: 'confetti',
                x: centerX + (Math.random() - 0.5) * 150, // Start from wider area
                y: centerY - 50 + (Math.random() - 0.5) * 100, // Start slightly above
                vx: Math.cos(angle) * velocity * 0.8, // Slower horizontal
                vy: Math.sin(angle) * velocity * 0.5 + 1, // Fall down with slight spread
                color: colors[Math.floor(Math.random() * colors.length)],
                size: size,
                rotation: Math.random() * Math.PI * 2,
                rotationSpeed: (Math.random() - 0.5) * 0.3, // More rotation
                life: 1.0,
                decay: 0.01 + Math.random() * 0.008, // Slower decay
                shape: Math.random() > 0.3 ? 'rect' : 'circle', // More rectangles
                gravity: 0.15 // Light gravity for confetti
            };
        }
        
        /**
         * Create a particle for explosion effect
         */
        createParticle(centerX, centerY) {
            const colors = this.options.celebration.colors || ['#667eea', '#764ba2', '#f093fb', '#4facfe'];
            const angle = (Math.PI * 2 * Math.random());
            // Particles: fast explosion from center, smaller, sparkle effect
            const velocity = 4 + Math.random() * 6; // Faster explosion
            const size = 2 + Math.random() * 4; // Smaller particles
            
            return {
                type: 'particle',
                x: centerX,
                y: centerY, // Start exactly from center
                vx: Math.cos(angle) * velocity,
                vy: Math.sin(angle) * velocity,
                color: colors[Math.floor(Math.random() * colors.length)],
                size: size,
                life: 1.0,
                decay: 0.025 + Math.random() * 0.015, // Faster decay
                gravity: 0.2, // Stronger gravity
                glow: true // Add glow effect for particles
            };
        }
        
        /**
         * Animate celebration particles
         */
        animateCelebration() {
            if (!this.isCelebrating || !this.celebrationCtx) return;
            
            const dpr = window.devicePixelRatio || 1;
            const width = this.canvas.width / dpr;
            const height = this.canvas.height / dpr;
            
            // Clear canvas
            this.celebrationCtx.clearRect(0, 0, width, height);
            
            // Update and draw particles
            const activeParticles = [];
            
            for (let i = 0; i < this.celebrationParticles.length; i++) {
                const p = this.celebrationParticles[i];
                
                // Update position
                p.x += p.vx;
                p.y += p.vy;
                
                // Apply gravity - different for confetti and particles
                if (p.gravity) {
                    if (p.type === 'particle') {
                        // Strong gravity for particles (sparkles fall down)
                        p.vy += p.gravity;
                    } else if (p.type === 'confetti') {
                        // Light gravity for confetti (float down slowly)
                        p.vy += p.gravity * 0.5;
                    }
                }
                
                // Update rotation for confetti (more visible)
                if (p.type === 'confetti' && p.rotationSpeed) {
                    p.rotation += p.rotationSpeed;
                }
                
                // Update life
                p.life -= p.decay;
                
                // Keep particle if still alive and in bounds
                if (p.life > 0 && p.x > -50 && p.x < width + 50 && p.y > -50 && p.y < height + 50) {
                    activeParticles.push(p);
                    
                    // Draw particle with different styles for confetti vs particles
                    this.celebrationCtx.save();
                    this.celebrationCtx.globalAlpha = p.life;
                    
                    if (p.type === 'confetti') {
                        // Confetti: colorful rectangles and circles, more visible
                        this.celebrationCtx.fillStyle = p.color;
                        this.celebrationCtx.translate(p.x, p.y);
                        this.celebrationCtx.rotate(p.rotation);
                        
                        if (p.shape === 'rect') {
                            // Rectangular confetti pieces
                            const rectWidth = p.size;
                            const rectHeight = p.size * 0.6; // Slightly rectangular
                            this.celebrationCtx.fillRect(-rectWidth / 2, -rectHeight / 2, rectWidth, rectHeight);
                        } else {
                            // Circular confetti pieces
                            this.celebrationCtx.beginPath();
                            this.celebrationCtx.arc(0, 0, p.size / 2, 0, Math.PI * 2);
                            this.celebrationCtx.fill();
                        }
                    } else {
                        // Particles: small glowing sparkles with glow effect
                        if (p.glow) {
                            // Add glow effect for particles
                            this.celebrationCtx.shadowBlur = 8;
                            this.celebrationCtx.shadowColor = p.color;
                        }
                        this.celebrationCtx.fillStyle = p.color;
                        this.celebrationCtx.beginPath();
                        this.celebrationCtx.arc(p.x, p.y, p.size, 0, Math.PI * 2);
                        this.celebrationCtx.fill();
                        
                        // Add inner bright spot for sparkle effect
                        this.celebrationCtx.globalAlpha = p.life * 0.6;
                        this.celebrationCtx.fillStyle = '#ffffff';
                        this.celebrationCtx.beginPath();
                        this.celebrationCtx.arc(p.x, p.y, p.size * 0.4, 0, Math.PI * 2);
                        this.celebrationCtx.fill();
                    }
                    
                    this.celebrationCtx.restore();
                }
            }
            
            this.celebrationParticles = activeParticles;
            
            // Continue animation if there are active particles
            if (this.celebrationParticles.length > 0) {
                this.celebrationAnimationFrame = requestAnimationFrame(() => this.animateCelebration());
            } else {
                // Animation complete
                this.stopCelebration();
            }
        }
        
        /**
         * Stop celebration animation
         */
        stopCelebration() {
            this.isCelebrating = false;
            
            if (this.celebrationAnimationFrame) {
                cancelAnimationFrame(this.celebrationAnimationFrame);
                this.celebrationAnimationFrame = null;
            }
            
            // Remove canvas after fade out
            if (this.celebrationCanvas) {
                setTimeout(() => {
                    if (this.celebrationCanvas && this.celebrationCanvas.parentNode) {
                        this.celebrationCanvas.parentNode.removeChild(this.celebrationCanvas);
                    }
                    this.celebrationCanvas = null;
                    this.celebrationCtx = null;
                }, 500);
            }
            
            this.celebrationParticles = [];
        }

        /**
         * Update background pattern (when options change)
         */
        updateBackgroundPatternOptions() {
            if (this.options.backgroundPattern && this.options.backgroundPattern.enabled) {
                if (!this.patternCanvas) {
                    this.createBackgroundPattern();
                } else {
                    this.updateBackgroundPattern();
                }
            }
        }


        /**
         * Destroy the timer instance
         */
        /**
         * Clean up localStorage entry for this instance
         * Called when destroying timer to prevent counting old instances
         */
        cleanupLocalStorageSync() {
            if (!this.realTimeSync.enabled || !this.realTimeSync.roomId) return;
            
            const roomKey = 'countdown_timer_sync_' + this.realTimeSync.roomId;
            
            try {
                // Remove all entries for this instance ID
                for (let i = localStorage.length - 1; i >= 0; i--) {
                    const key = localStorage.key(i);
                    if (key && key.startsWith(roomKey + '_')) {
                        try {
                            const data = JSON.parse(localStorage.getItem(key));
                            if (data && data.instanceId === this.instanceId) {
                                localStorage.removeItem(key);
                            }
                        } catch (e) {
                            // Ignore parse errors, but still try to remove if key matches instanceId pattern
                            if (key.includes(this.instanceId)) {
                                localStorage.removeItem(key);
                            }
                        }
                    }
                }
            } catch (e) {
                // Ignore errors during cleanup
            }
        }
        
        destroy() {
            // Stop all animations first
            this.stop();
            this.stopCelebration();
            
            // Clear all timeouts and intervals
            if (this.resizeTimeout) {
                clearTimeout(this.resizeTimeout);
                this.resizeTimeout = null;
            }
            
            // CRITICAL: Clean up localStorage entry BEFORE disconnecting
            // This prevents old instances from being counted as active viewers
            if (this.realTimeSync.enabled) {
                this.cleanupLocalStorageSync();
                this.disconnect();
            }
            
            // Clean up AI optimization intervals
            if (this.aiOptimization && this.aiOptimization.optimizationIntervalId) {
                clearInterval(this.aiOptimization.optimizationIntervalId);
                this.aiOptimization.optimizationIntervalId = null;
            }
            
            // Remove event listeners
            $(window).off('resize.' + this.instanceId);
            
            // Disconnect observers
            if (this.resizeObserver) {
                this.resizeObserver.disconnect();
                this.resizeObserver = null;
            }
            
            if (this.mutationObserver) {
                this.mutationObserver.disconnect();
                this.mutationObserver = null;
            }
            
            // Clear canvas references
            if (this.canvas) {
                const ctx = this.canvas.getContext('2d');
                if (ctx) {
                    ctx.clearRect(0, 0, this.canvas.width, this.canvas.height);
                }
            }
            
            // Remove canvas from DOM
            if (this.canvas && this.canvas.parentNode) {
                this.canvas.parentNode.removeChild(this.canvas);
            }
            
            // Clear all references
            this.canvas = null;
            this.ctx = null;
            this.animationFrame = null;
            
            // Reset timer state
            this.startTime = null;
            this.endTime = null;
            if (this.mutationObserver) {
                this.mutationObserver.disconnect();
            }
            this.$element.empty();
        }
    }

    /**
     * Default options
     */
    CountdownTimerPro.defaults = {
        // Timer type: 'countdown' or 'countup'
        timeType: 'countdown',
        
        // Time source: 'client' or 'server'
        timeSource: 'client',
        
        // For countdown with client source: duration in seconds
        duration: 3600,
        
        // For countdown with server source: Unix timestamp (seconds)
        endTime: null,
        
        // For countup: start Unix timestamp (seconds), null for current time
        startTime: null,
        
        // Display options
        showDays: true,
        showHours: true,
        showMinutes: true,
        showSeconds: true,
        
        // Size options
        width: 800,
        height: 400,
        fontSize: 48,
        circleThickness: 8,
        
        // Color options
        colors: {
            circle: '#3498db',
            circleBackground: '#ecf0f1',
            text: '#2c3e50'
        },
        
        // Font options
        fontFamily: 'Arial, sans-serif',
        fontWeight: 'bold', // 'normal', 'bold', 'lighter', 'bolder', or numeric '100'-'900'
        labelFontWeight: 'normal', // Font weight for labels
        
        // Multicolor options
        multicolor: {
            enabled: false,
            colors: ['#3498db', '#e74c3c', '#2ecc71', '#f39c12']
        },
        
        // Pulse/Glow animation
        pulseGlow: {
            enabled: false,
            intensity: 1.0, // 0.5 to 2.0
            speed: 1.0 // 0.5 to 2.0
        },
        
        // Gradient Wave animation
        gradientWave: {
            enabled: false,
            colors: ['#667eea', '#764ba2', '#f093fb', '#4facfe'],
            speed: 1.0 // 0.5 to 2.0
        },
        
        // Sound effects
        sound: {
            enabled: false,
            tickEnabled: false, // Sound every second
            tickVolume: 0.3, // 0.0 to 1.0
            finishEnabled: true, // Sound when timer finishes
            finishVolume: 0.5, // 0.0 to 1.0
            tickFrequency: 800, // Hz for tick sound
            finishFrequency: 600 // Hz for finish sound
        },
        
        // 3D Effects
        effects3D: {
            enabled: false,
            shadowDepth: 20, // 0 to 50
            rotationSpeed: 0, // 0 = no rotation, 0.01 to 0.1 for rotation
            parallaxIntensity: 0, // 0 to 1
            lightingIntensity: 0.5 // 0 to 1
        },
        
        // Liquid Effect (Distinctive visual feature - Liquid that evaporates with bubbles)
        // Always enabled by default - High definition liquid effect
        liquidEffect: {
            enabled: true, // Always enabled - distinctive visual feature
            intensity: 1.2, // High definition - more bubbles and smoother waves
            bubbleSpeed: 1.0, // 0.5 to 2.0 - Speed of bubbles rising
            waveSpeed: 1.0, // 0.5 to 2.0 - Speed of surface waves
            bubbleSpawnRate: 0.8, // Higher rate for more realistic effect
            showEvaporation: true, // Show evaporation effect at surface
            liquidColor: null // null = use circle color, or specify custom color
        },
        
        // Background Pattern
        backgroundPattern: {
            enabled: false,
            type: 'grid', // 'grid', 'dots', 'lines', 'diagonal', 'none'
            color: 'rgba(102, 126, 234, 0.1)',
            size: 20, // Pattern size in pixels
            spacing: 10 // Spacing between pattern elements
        },
        
        // Text labels
        texts: {
            days: 'DAYS',
            hours: 'HOURS',
            minutes: 'MINUTES',
            seconds: 'SECONDS'
        },
        
        // Expire action: 'hide', 'redirect', 'message', or 'none'
        expireAction: 'none',
        expireMessage: 'Time is up!',
        expireRedirectUrl: '',
        
        // Auto reset options
        autoReset: {
            enabled: false,
            unit: 'minutes', // 'minutes', 'hours', 'days'
            value: 1
        },
        
        // Callback functions
        onFinish: null,
        
        // Celebration animation options
        celebration: {
            enabled: false,
            type: 'both', // 'confetti', 'particles', or 'both'
            intensity: 1.0, // 0.5 to 2.0 (multiplier for particle count)
            duration: 3000, // Animation duration in ms
            colors: ['#667eea', '#764ba2', '#f093fb', '#4facfe', '#43e97b', '#fa709a', '#fee140', '#30cfd0']
        },
        
        // Cross-tab synchronization with optional WebSocket server support (FEATURE 1)
        realTimeSync: {
            enabled: false,
            roomId: null, // Auto-generated if null (format: 'room_xxxxx')
            serverUrl: null, // WebSocket server URL (optional, uses localStorage fallback if null)
            showViewerCount: true, // Show "X people watching" counter (default: enabled)
            showConnectionStatus: false, // Show connection status indicator (disabled by default)
            syncInterval: 1000, // Sync every 1 second (ms)
            reconnectInterval: 3000, // Reconnect after 3 seconds if disconnected (ms)
            maxReconnectAttempts: 5, // Max reconnection attempts
            useLocalStorageFallback: true // Use localStorage for sync if WebSocket unavailable
        },
        
        // Behavioral targeting (FEATURE 2: UNIQUE - Timer che cambia in base a comportamento utente)
        behavioralTargeting: {
            enabled: false,
            rules: [], // Array of rules: [{condition: 'newVisitor', duration: 86400, message: '...'}, ...]
            priority: ['newVisitor', 'returningVisitor', 'device', 'source'], // Order of rule evaluation
            storageKey: 'countdown_timer_visits', // localStorage key for tracking visits
            visitThreshold: 2 // Number of visits to be considered "returning visitor"
        },
        
        // Conversion Optimization (FEATURE 3: Local A/B testing for automatic optimization)
        aiOptimization: {
            enabled: false,
            trackInteractions: true, // Track clicks, views, etc.
            trackConversions: true, // Track conversions (call trackConversion() manually)
            learningRate: 0.1, // How fast the AI adapts (0.0 to 1.0)
            minSamples: 10, // Minimum interactions before making changes
            optimizationInterval: 3600000, // Check every hour (ms)
            storageKey: 'countdown_timer_ai_data' // localStorage key for AI data
        }
    };

    /**
     * jQuery plugin definition
     */
    $.fn.countdownTimerPro = function(options) {
        return this.each(function() {
            const $this = $(this);
            let instance = $this.data('countdownTimerPro');
            
            // Handle method calls
            if (typeof options === 'string') {
                if (instance && typeof instance[options] === 'function') {
                    instance[options]();
                }
                return $this;
            }
            
            // Initialize new instance
            if (!instance) {
                instance = new CountdownTimerPro(this, options);
                $this.data('countdownTimerPro', instance);
            }
            
            return $this;
        });
    };

    // Auto-initialize on DOM ready
    $(document).ready(function() {
        $('[data-countdown-timer-pro]').each(function() {
            const $this = $(this);
            const options = $this.data('countdown-timer-pro-options') || {};
            $this.countdownTimerPro(options);
        });
    });

})(jQuery);

